const SERVER_URL = "https://idmanager.site";
const API_ENDPOINTS = {
    PING: `${SERVER_URL}/api/ids/ping`,
    PING_FAST: `${SERVER_URL}/api/ids/ping-fast`,
    VALIDATE_KEY: `${SERVER_URL}/api/extension/validate-key`,
    PARSERS: `${SERVER_URL}/api/parsers/parsers`,
    PARSERS_EXPORT: `${SERVER_URL}/api/parsers/parsers/export`,
    SITE_SETTINGS: `${SERVER_URL}/api/sites/site-settings`,
    SITE_GROUPS: `${SERVER_URL}/api/sites/site-groups`
};

// Helper: proxy requests through background to attach API key and avoid CORS
function apiRequest({ url, method = 'GET', body = null, headers = {} } = {}) {
    return new Promise((resolve) => {
        chrome.runtime.sendMessage({ action: 'apiRequest', url, method, body, headers }, (resp) => {
            resolve(resp || { status: 0, error: 'no-response' });
        });
    });
}

async function initPopup() {
    // Mark executed for debug-runner
    window.__popupInitExecuted = true;
    const debugEl = document.getElementById("popupDebug");

    // Global error handlers to catch runtime problems in popup context
    window.addEventListener('error', (ev) => {
        try {
            const msg = ev && ev.message ? ev.message : String(ev);
            console.error('Global error:', ev);
            if (debugEl) debugEl.textContent = 'Ошибка: ' + msg;
        } catch (e) {}
    });

    window.addEventListener('unhandledrejection', (ev) => {
        try {
            console.error('Unhandled rejection:', ev.reason);
            if (debugEl) debugEl.textContent = 'Unhandled rejection: ' + (ev.reason && ev.reason.message ? ev.reason.message : String(ev.reason));
        } catch (e) {}
    });

    try {
        if (debugEl) debugEl.textContent = 'JS запущен — инициализация...';

        const input = document.getElementById("apiKeyInput");
        const openHelpBtn = document.getElementById('openServerHelp');
        const saveBtn = document.getElementById("saveKey");
    const clearBtn = document.getElementById("clearKey");
    const checkBtn = document.getElementById("checkServer");
    const refreshBtn = document.getElementById("refreshSettings");
    const enabledToggle = document.getElementById("enabledToggle");
    const rescanBtn = document.getElementById("rescanPage");
    const keyStatusEl = document.getElementById("keyStatus");
    const serverStatusEl = document.getElementById("serverStatus");
    const settingsEl = document.getElementById("currentSettings");
    const parsersListEl = document.getElementById("parsersList");
    const parsersCountEl = document.getElementById("parsersCount");
    const groupsListEl = document.getElementById("groupsList");
    const parsersTabs = document.getElementById("parsersTabs");
    const refreshParsersBtn = document.getElementById("refreshParsersBtn");
    const selectedGroupNameEl = document.getElementById("selectedGroupName");
    const parserSearchInput = document.getElementById("parserSearchInput");
    const pickSelectorBtn = document.getElementById("pickSelectorBtn");
    const pickedSelectorInput = document.getElementById("pickedSelector");
    const copySelectorBtn = document.getElementById("copySelectorBtn");

    chrome.runtime.onMessage.addListener((request) => {
        if (request?.action !== 'subscriptionExpired') return;
        const err = request?.details?.error || '';
        let message = 'Подписка истекла. Продлите доступ.';
        if (err === 'subscription_grace_expired') {
            message = 'Подписка истекла более 7 дней назад. Доступ закрыт.';
        } else if (err === 'subscription_expired_admin_add_disabled') {
            message = 'Подписка истекла. Добавление ID недоступно.';
        }
        if (keyStatusEl) {
            keyStatusEl.textContent = `${message}`;
            keyStatusEl.style.color = '#991b1b';
        }
    });

    if (openHelpBtn) {
        openHelpBtn.addEventListener('click', () => {
            try {
                chrome.tabs.create({ url: `${SERVER_URL}/#help` });
            } catch (e) {
                console.warn('Не удалось открыть вкладку с инструкцией', e);
            }
        });
    }

    let currentParsers = [];
    let currentSite = "";
    // Track current view to avoid overwriting group selection when content script pushes updates
    let currentView = { type: 'site', groupName: null }; // types: 'site'|'all'|'groups'|'group' 

    let parserSearchTerm = '';
    let pickerActive = false;

    // Keep popup compact: show only first N parsers until user expands
    const PARSERS_DEFAULT_LIMIT = 6;
    let parsersExpanded = false;

    function updateParsersCount(visibleCount, totalCount) {
        if (!parsersCountEl) return;
        if (typeof totalCount !== 'number') {
            parsersCountEl.textContent = String(visibleCount ?? 0);
            return;
        }
        if (parserSearchTerm && visibleCount !== totalCount) {
            parsersCountEl.textContent = `${visibleCount}/${totalCount}`;
        } else {
            parsersCountEl.textContent = String(totalCount);
        }
    }

    function filterParsers(list) {
        if (!parserSearchTerm) return list;
        const q = parserSearchTerm.toLowerCase();
        return (list || []).filter((p) => {
            const name = (p.name || '').toLowerCase();
            const site = (p.site || '').toLowerCase();
            const type = (p.type || p.pattern_type || '').toLowerCase();
            const pattern = (p.pattern || p.code || '').toLowerCase();
            const desc = (p.description || '').toLowerCase();
            return name.includes(q) || site.includes(q) || type.includes(q) || pattern.includes(q) || desc.includes(q);
        });
    }

    // ==================== BULK API ДЛЯ ID ====================
    async function bulkCheckIds(ids) {
        if (!Array.isArray(ids) || ids.length === 0) return {};
        try {
            const res = await apiRequest({ url: `${SERVER_URL}/api/ids/bulk-check-ids`, method: 'POST', body: { ids, site: currentSite } });
            if (res.status === 200) return (res.data && res.data.results) || {};
        } catch (e) {
            console.error("Ошибка пакетной проверки ID:", e);
        }
        return {};
    }

    async function bulkAddIds(ids, apiKey) {
        if (!Array.isArray(ids) || ids.length === 0) return false;
        try {
            const res = await apiRequest({ url: `${SERVER_URL}/api/ids/bulk-add`, method: 'POST', body: { ids, apiKey, site: currentSite } });
            if (res.status === 200) return res.data && res.data.success === true;
        } catch (e) {
            console.error("Ошибка пакетной отправки ID:", e);
        }
        return false;
    }

    // Загрузка текущего ключа и настроек
    if (debugEl) debugEl.textContent = 'Инициализация: чтение локального storage...';
    const hasKey = await loadInitialData();
    let awaitingKey = false;
    if (!hasKey) {
        awaitingKey = true;
        if (debugEl) debugEl.textContent = 'Ожидание API-ключа...';
        // Don't load groups/parsers until user provides API key
    }
    if (debugEl) debugEl.textContent = 'Инициализация: загрузка групп...';
    // ensure groups fetched even if loadInitialData already did - refresh state
    try { if (!awaitingKey) await loadGroups(); } catch (e) { console.warn('Ошибка loadGroups при инициализации:', e); }
    if (debugEl) debugEl.textContent = 'Инициализация: загрузка парсеров...';
    // If current view is 'all' or 'group' call appropriate loader
    try {
        if (!awaitingKey) {
            if (currentView && currentView.type === 'all') await loadAllParsers();
            else if (currentView && currentView.type === 'group' && currentView.groupName) await loadGroupParsers(currentView.groupName);
            else await loadParsersAndSettings();
        }
    } catch (e) {
        console.warn('Ошибка при начальной загрузке парсеров:', e);
    }
    if (debugEl) debugEl.textContent = 'Инициализация: завершена';

    // Загрузка начальных данных
    async function loadInitialData() {
        try {
            // Загружаем API ключ
            const result = await new Promise(resolve => {
                chrome.storage.local.get(["apiKey", "my_api_key", "enabled"], resolve);
            });
            
            input.value = result.apiKey || result.my_api_key || "";
            enabledToggle.checked = result.enabled !== false;
            // If no key, do not load remote data
            if (!input.value) {
                showStatus(keyStatusEl, "Введите API-ключ для загрузки настроек", "warning");
                return false;
            }

            // Загружаем информацию о парсерах и настройках
            await loadParsersAndSettings();
            // Загружаем группы для вкладки Groups
            await loadGroups();

            if (input.value) {
                showStatus(keyStatusEl, "Ключ загружен", "success");
            }
            return true;
        } catch (error) {
            console.error("Ошибка загрузки данных:", error);
            showStatus(keyStatusEl, "Ошибка загрузки", "error");
        }
    }

    // Загрузка парсеров и настроек
    // Улучшенная загрузка с fallback механизмом
async function loadParsersAndSettings() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab || !tab.id || !tab.url) {
            // No active tab or unsupported URL (chrome:// etc) - fall back to 'default' (show global parsers)
            currentSite = 'default';
        } else {
            try {
                const url = new URL(tab.url);
                currentSite = url.hostname.replace(/^(www\.)/, '');
            } catch (err) {
                currentSite = 'default';
            }
        }

        // Пробуем экспорт парсеров с таймаутом
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 5000);
        
        try {
            const response = await apiRequest({ url: `${API_ENDPOINTS.PARSERS_EXPORT}?site=${encodeURIComponent(currentSite)}` });
            clearTimeout(timeoutId);
            if (response.status === 200) {
                const data = response.data;
                if (data && data.success) {
                    currentParsers = data.parsers || [];
                    displayParsers(currentParsers);
                    return;
                }
            }
        } catch (exportError) {
            clearTimeout(timeoutId);
            console.log('Экспорт парсеров не доступен, используем обычный...');
        }

        // Fallback на обычный эндпоинт
        try {
            const response = await apiRequest({ url: `${API_ENDPOINTS.PARSERS}?site=${encodeURIComponent(currentSite)}` });
            if (response.status === 200) {
                const parsers = response.data;
                currentParsers = Array.isArray(parsers) ? parsers.filter(p => p.is_active !== false) : (parsers.parsers || []);
                displayParsers(currentParsers);
            }
        } catch (e) {
            console.warn('Ошибка при получении парсеров', e);
        }
        
        // Загружаем настройки сайта
        try {
            const settingsResponse = await apiRequest({ url: `${API_ENDPOINTS.SITE_SETTINGS}?site=${encodeURIComponent(currentSite)}` });
            if (settingsResponse.status === 200) {
                const settings = settingsResponse.data;
                displayCurrentSettings(settings, currentSite);
            }
        } catch (e) {
            console.warn('Ошибка при получении настроек сайта', e);
        }
        
    } catch (error) {
        console.error("Ошибка загрузки:", error);
        parsersListEl.innerHTML = `<div class="no-parsers">Ошибка загрузки парсеров: ${escapeHtml(error.message || 'unknown')}</div>`;
    }
}

    // Прямая загрузка с сервера (через background proxy)
    async function loadFromServerDirectly() {
        try {
            const response = await apiRequest({ url: `${SERVER_URL}/api/parsers/export?site=${encodeURIComponent(currentSite)}` });
            if (response.status !== 200) throw new Error(`HTTP ${response.status}`);
            const data = response.data;

            if (data && data.success && data.parsers) {
                currentParsers = data.parsers;
                displayParsers(currentParsers);
            } else {
                const fallbackResponse = await apiRequest({ url: `${SERVER_URL}/api/parsers?site=${encodeURIComponent(currentSite)}` });
                if (fallbackResponse.status === 200) {
                    const parsers = fallbackResponse.data;
                    currentParsers = Array.isArray(parsers) ? parsers.filter(parser => parser.is_active !== false) : (parsers.parsers || []);
                    displayParsers(currentParsers);
                }
            }

            const settingsResponse = await apiRequest({ url: `${SERVER_URL}/api/site-settings?site=${encodeURIComponent(currentSite)}` });
            if (settingsResponse.status === 200) {
                const settings = settingsResponse.data;
                displayCurrentSettings(settings, currentSite);
            } else {
                displayDefaultSettings();
            }

        } catch (error) {
            console.error("Ошибка прямой загрузки:", error);
            showFallbackState();
        }
    }

    // Load ALL parsers (no site filter)
    async function loadAllParsers() {
        try {
            const resp = await apiRequest({ url: API_ENDPOINTS.PARSERS });
            if (resp.status !== 200) throw new Error('HTTP ' + resp.status);
            const data = resp.data;
            const list = Array.isArray(data) ? data : (data.parsers || []);
            currentParsers = list.filter(p => p.is_active !== false);
            displayParsers(currentParsers);
        } catch (err) {
            console.warn('Ошибка загрузки всех парсеров:', err);
            parsersListEl.innerHTML = '<div class="no-parsers">Ошибка загрузки парсеров</div>';
        }
    }

    // Загрузка списка групп сайтов
    async function loadGroups() {
        try {
            groupsListEl.innerHTML = '<div class="loading"><div class="loading-spinner"></div>Загрузка групп...</div>';
            const resp = await apiRequest({ url: API_ENDPOINTS.SITE_GROUPS });
            if (resp.status !== 200) {
                groupsListEl.innerHTML = '<div class="no-parsers">Ошибка загрузки групп</div>';
                return [];
            }
            const data = resp.data;
            const rawList = Array.isArray(data) ? data : (data.groups || data.siteGroups || []);
            // Normalize items: support both string names and objects { group_name, name }
            const list = (rawList || []).map(item => {
                if (typeof item === 'string') return item;
                if (!item) return '';
                return item.group_name || item.name || item.group || '';
            }).filter(Boolean);

            if (!Array.isArray(list) || list.length === 0) {
                groupsListEl.innerHTML = '<div class="no-parsers">Нет групп</div>';
                return [];
            }

            // render group tags
            groupsListEl.innerHTML = list.map(g => `<button class="group-tag" data-group-name="${escapeHtml(g)}">${escapeHtml(g)}</button>`).join('');
            return list;
        } catch (err) {
            console.warn('Ошибка loadGroups:', err);
            groupsListEl.innerHTML = '<div class="no-parsers">Ошибка загрузки групп</div>';
            return [];
        }
    }

    // Показать состояние fallback
    function showFallbackState() {
        parsersListEl.innerHTML = '<div class="no-parsers">Ошибка загрузки парсеров</div>';
        updateParsersCount(0, 0);
        displayDefaultSettings();
    }

    // Показать настройки по умолчанию
    function displayDefaultSettings() {
        settingsEl.innerHTML = `
            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Настройки не загружены</div>
                    <div class="setting-value">Обновите страницу или проверьте сервер</div>
                </div>
            </div>
        `;
    }

    // Функция отображения статуса
    function showStatus(element, message, type) {
        element.textContent = message;
        element.className = `status ${type}`;
        element.style.display = 'block';
        setTimeout(() => {
            element.style.display = 'none';
        }, 5000);
    }

    // Отображение списка парсеров (улучшенная версия с действиями)
    async function displayParsers(parsers) {
        const total = Array.isArray(parsers) ? parsers.length : 0;
        const filtered = filterParsers(Array.isArray(parsers) ? parsers : []);
        updateParsersCount(filtered.length, total);

        const hasMore = !parsersExpanded && filtered.length > PARSERS_DEFAULT_LIMIT;
        const visible = parsersExpanded ? filtered : filtered.slice(0, PARSERS_DEFAULT_LIMIT);

        if (!filtered || filtered.length === 0) {
            if (parserSearchTerm) {
                parsersListEl.innerHTML = '<div class="no-parsers">Ничего не найдено по запросу</div>';
            } else {
                parsersListEl.innerHTML = '<div class="no-parsers">Нет активных парсеров для этого сайта</div>';
            }
            return;
        }

        const ds = await new Promise(r => chrome.storage.local.get(['disabledParsers'], r));
        const disabledSet = new Set((ds.disabledParsers || []).map(x => Number(x)));

        const html = visible.map(parser => {
            const disabled = disabledSet.has(Number(parser.id));
            return `
            <div class="parser-item ${disabled ? 'parser-disabled' : ''}" data-parser-id="${parser.id}">
                <div class="parser-card">
                    <div class="parser-row" style="display:flex; justify-content:space-between; align-items:flex-start; gap:8px; flex-wrap:wrap;">
                        <div class="parser-main" style="min-width: 0; flex: 1 1 240px;">
                            <div class="parser-name">${escapeHtml(parser.name)}</div>
                            <div class="parser-meta">Тип: ${parser.type || parser.pattern_type || 'auto'} • Сайт: ${parser.site}</div>
                        </div>
                        <div class="parser-actions" style="flex: 0 0 auto; display:flex; align-items:center; gap:6px; flex-wrap:wrap; justify-content:flex-end;">
                            <button class="parser-action-btn btn-copy-pattern" title="Copy pattern">📋</button>
                            <button class="parser-action-btn btn-copy-selectors" title="Copy selectors">🧩</button>
                            <button class="parser-action-btn btn-test-parser" title="Test parser">🧪</button>
                            <button class="parser-action-btn btn-highlight" title="Highlight on page">✨</button>
                            <div class="parser-toggle">
                                <label class="switch small" title="${disabled ? 'Включить парсер' : 'Отключить парсер'}">
                                    <input type="checkbox" class="parser-disabled-checkbox" data-parser-id="${parser.id}" ${disabled ? 'checked' : ''} aria-label="Toggle parser ${escapeHtml(parser.name)}">
                                    <span class="slider"></span>
                                </label>
                                <button class="toggle-chip ${disabled ? 'disabled' : 'enabled'}" data-parser-id="${parser.id}" aria-pressed="${disabled ? 'true' : 'false'}">${disabled ? '⛔ Отключён' : '✅ Включён'}</button>
                            </div>
                        </div>
                    </div>
                    ${parser.description ? `<div style="font-size:11px; color:#6c757d;">${escapeHtml(parser.description)}</div>` : ''}
                    <div class="parser-pattern" style="font-size: 11px; color: #adb5bd; font-family: monospace; word-break: break-word; overflow-wrap: anywhere;">${escapeHtml(parser.pattern).substring(0, 160)}${parser.pattern && parser.pattern.length > 160 ? '...' : ''}</div>
                </div>
            </div>`;
        }).join('');

        const footer = (filtered.length > PARSERS_DEFAULT_LIMIT)
            ? `<div class="list-footer">
                  ${parsersExpanded
                      ? `<button id="parsersShowLessBtn" class="btn small" type="button">Свернуть</button>`
                      : `<button id="parsersShowMoreBtn" class="btn small" type="button">Показать ещё (${filtered.length - PARSERS_DEFAULT_LIMIT})</button>`}
               </div>`
            : '';

        parsersListEl.innerHTML = html + footer;
        return;
    }

    // Load parsers for a specific group (returns parser objects)
    async function loadGroupParsers(groupName) {
        try {
            const res = await apiRequest({ url: `${API_ENDPOINTS.SITE_GROUPS}/${encodeURIComponent(groupName)}/parsers` });
            if (res.status !== 200) throw new Error('HTTP ' + res.status);
            const data = res.data;
            const ids = data.parsers || [];
            // Fetch all parsers and filter
            const allRes = await apiRequest({ url: API_ENDPOINTS.PARSERS });
            if (allRes.status !== 200) throw new Error('HTTP ' + allRes.status);
            const allParsers = allRes.data;
            const list = Array.isArray(allParsers) ? allParsers : (allParsers.parsers || []);
            const idSet = new Set(ids.map(x => Number(x)));
            const parsers = list.filter(p => idSet.has(Number(p.id)));
            currentParsers = parsers;
            displayParsers(parsers);
        } catch (err) {
            console.warn('Ошибка загрузки парсеров группы:', err);
            parsersListEl.innerHTML = '<div class="no-parsers">Ошибка загрузки парсеров группы</div>';
        }
    }

    // Set local disabled state for parser (explicit)
    async function setLocalDisable(parserId, disabled) {
        const data = await new Promise(r => chrome.storage.local.get(['disabledParsers'], r));
        const set = new Set((data.disabledParsers || []).map(x => Number(x)));
        if (disabled) set.add(Number(parserId)); else set.delete(Number(parserId));
        await chrome.storage.local.set({ disabledParsers: Array.from(set) });

        // Notify content script to refresh
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab && tab.id) {
                await chrome.tabs.sendMessage(tab.id, { action: 'refreshParsersFromPopup' });
            }
        } catch (err) {
            console.warn('Не удалось отправить сообщение content script:', err);
        }

        // Update UI without full reload
        const item = parsersListEl.querySelector(`[data-parser-id="${parserId}"]`);
        if (item) {
            if (disabled) {
                item.classList.add('parser-disabled');
                const tl = item.querySelector('.toggle-label-small');
                if (tl) tl.textContent = 'Отключён';
                const cbNode = item.querySelector('.parser-disabled-checkbox');
                if (cbNode) cbNode.checked = true;
            } else {
                item.classList.remove('parser-disabled');
                const tl = item.querySelector('.toggle-label-small');
                if (tl) tl.textContent = 'Включён';
                const cbNode = item.querySelector('.parser-disabled-checkbox');
                if (cbNode) cbNode.checked = false;
            }
        }
    }

    // Copy to clipboard helpers
    function copyToClipboard(text) {
        navigator.clipboard.writeText(text).catch(() => {});
    }

    // Test parser by sending request to server
    async function testParser(pattern, site) {
        try {
            const res = await apiRequest({ url: `${SERVER_URL}/api/parsers/test-parser`, method: 'POST', body: { code: pattern, site: site || '' } });
            if (res.status === 200) {
                const data = res.data;
                showStatus(serverStatusEl, `Test result: ${data && data.success ? 'ok' : 'fail'}`, (data && data.success) ? 'success' : 'warning');
            } else {
                showStatus(serverStatusEl, `Test failed (${res.status})`, 'warning');
            }
        } catch (err) {
            console.warn('Ошибка теста парсера:', err);
            showStatus(serverStatusEl, 'Ошибка тестирования', 'error');
        }
    }

    function sendTabMessage(tabId, message, timeoutMs = 1200) {
        return new Promise((resolve) => {
            let settled = false;
            const timer = setTimeout(() => {
                if (settled) return;
                settled = true;
                resolve({ success: false, error: 'timeout' });
            }, timeoutMs);
            try {
                chrome.tabs.sendMessage(tabId, message, (resp) => {
                    if (settled) return;
                    settled = true;
                    clearTimeout(timer);
                    const err = chrome.runtime.lastError;
                    if (err) {
                        resolve({ success: false, error: err.message || 'message error' });
                    } else {
                        resolve(resp || { success: true });
                    }
                });
            } catch (e) {
                if (settled) return;
                settled = true;
                clearTimeout(timer);
                resolve({ success: false, error: e.message || 'message error' });
            }
        });
    }

    // Highlight on page
    async function highlightOnPage(parserId) {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab || !tab.id) {
            showStatus(serverStatusEl, 'Нет активной вкладки', 'warning');
            return;
        }
        const resp = await sendTabMessage(tab.id, { action: 'highlightParser', parserId });
        if (!resp || resp.success === false) {
            showStatus(serverStatusEl, 'Подсветка не сработала', 'warning');
            return;
        }
        if (typeof resp.matched === 'number') {
            showStatus(serverStatusEl, `Подсвечено: ${resp.matched}`, resp.matched > 0 ? 'success' : 'warning');
        }
    }

    // Debounce
    function debounce(fn, wait) {
        let t;
        return (...args) => {
            clearTimeout(t);
            t = setTimeout(() => fn(...args), wait);
        };
    }

    function setPickerState(active) {
        pickerActive = active;
        if (!pickSelectorBtn) return;
        pickSelectorBtn.textContent = active ? '⛔ Отменить выбор' : '🎯 Выбрать элемент';
        if (active) {
            pickSelectorBtn.style.background = '#ffc107';
            pickSelectorBtn.style.color = '#000';
        } else {
            pickSelectorBtn.style.background = '';
            pickSelectorBtn.style.color = '';
        }
    }

    if (pickSelectorBtn) {
        pickSelectorBtn.addEventListener('click', async () => {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (!tab || !tab.id) {
                showStatus(serverStatusEl, 'Нет активной вкладки', 'warning');
                return;
            }
            const wantActive = !pickerActive;
            const action = wantActive ? 'startSelectorPicker' : 'stopSelectorPicker';
            const resp = await sendTabMessage(tab.id, { action });
            if (resp && resp.success !== false) {
                setPickerState(wantActive);
                if (wantActive) {
                    showStatus(serverStatusEl, 'Выберите элемент на странице', 'info');
                }
            } else {
                showStatus(serverStatusEl, 'Не удалось включить выбор', 'warning');
            }
        });
    }

    if (copySelectorBtn) {
        copySelectorBtn.addEventListener('click', () => {
            const val = pickedSelectorInput ? pickedSelectorInput.value.trim() : '';
            if (!val) {
                showStatus(serverStatusEl, 'Селектор не выбран', 'warning');
                return;
            }
            copyToClipboard(val);
            showStatus(serverStatusEl, 'Селектор скопирован', 'success');
        });
    }

    // Event delegation for parser actions + list controls
    parsersListEl.addEventListener('click', async (e) => {
        const moreBtn = e.target.closest('#parsersShowMoreBtn');
        if (moreBtn) {
            parsersExpanded = true;
            displayParsers(currentParsers);
            return;
        }

        const lessBtn = e.target.closest('#parsersShowLessBtn');
        if (lessBtn) {
            parsersExpanded = false;
            displayParsers(currentParsers);
            return;
        }

        const btn = e.target.closest('.parser-action-btn');
        if (!btn) return;
        const item = e.target.closest('.parser-item');
        if (!item) return;
        const parserId = item.getAttribute('data-parser-id');

        const patternNode = item.querySelector('.parser-pattern');
        const patternText = patternNode ? patternNode.textContent : '';

        if (btn.classList.contains('btn-copy-pattern')) {
            // Copy pattern
            copyToClipboard(patternText);
            showStatus(serverStatusEl, 'Pattern copied', 'success');
        } else if (btn.classList.contains('btn-copy-selectors')) {
            const parserIdNum = Number(parserId);
            const p = currentParsers.find(pp => Number(pp.id) === parserIdNum) || {};
            copyToClipboard((p.selectors || []).join('\n'));
            showStatus(serverStatusEl, 'Selectors copied', 'success');
        } else if (btn.classList.contains('btn-test-parser')) {
            const p = currentParsers.find(pp => Number(pp.id) === Number(parserId)) || {};
            await testParser(p.pattern || p.code || p.pattern || '', p.site);
        } else if (btn.classList.contains('btn-highlight')) {
            await highlightOnPage(Number(parserId));
        } else if (btn.classList.contains('toggle-chip')) {
            // Visible chip clicked - toggle state
            const chipParserId = btn.getAttribute('data-parser-id');
            const cb = item.querySelector(`input.parser-disabled-checkbox[data-parser-id="${chipParserId}"]`);
            const newDisabled = !cb.checked;
            cb.checked = newDisabled;
            await setLocalDisable(chipParserId, newDisabled);
        }
    });

    // Search/filter for parsers
    if (parserSearchInput) {
        const onSearch = debounce(() => {
            parserSearchTerm = (parserSearchInput.value || '').trim();
            parsersExpanded = false;
            displayParsers(currentParsers);
        }, 150);
        parserSearchInput.addEventListener('input', onSearch);
    }

    // Checkbox change handler for enable/disable
    parsersListEl.addEventListener('change', async (e) => {
        const cb = e.target.closest('.parser-disabled-checkbox');
        if (!cb) return;
        const parserId = cb.getAttribute('data-parser-id');
        const disabled = cb.checked;
        await setLocalDisable(parserId, disabled);
    });

    // Tabs handling (accessible)
    parsersTabs.addEventListener('click', (e) => {
        const tab = e.target.closest('.tab');
        if (!tab) return;
        // deactivate all
        parsersTabs.querySelectorAll('.tab').forEach(t => { t.classList.remove('active'); t.setAttribute('aria-selected','false'); t.tabIndex = -1; });
        // activate
        tab.classList.add('active');
        tab.setAttribute('aria-selected','true');
        tab.tabIndex = 0;

        const tabName = tab.getAttribute('data-tab');
        if (tabName === 'groups') {
            currentView = { type: 'groups', groupName: null };
            groupsListEl.style.display = 'block';
            if (selectedGroupNameEl) { selectedGroupNameEl.style.display = 'none'; selectedGroupNameEl.textContent = ''; }
            parsersExpanded = false;
            parsersListEl.innerHTML = '<div class="no-parsers">Выберите группу слева</div>';
        } else if (tabName === 'all') {
            currentView = { type: 'all', groupName: null };
            groupsListEl.style.display = 'none';
            if (selectedGroupNameEl) { selectedGroupNameEl.style.display = 'none'; selectedGroupNameEl.textContent = ''; }
            parsersExpanded = false;
            parsersListEl.innerHTML = '<div class="loading"><div class="loading-spinner"></div>Загрузка парсеров...</div>';
            loadAllParsers();
        } else {
            currentView = { type: 'site', groupName: null };
            groupsListEl.style.display = 'none';
            if (selectedGroupNameEl) { selectedGroupNameEl.style.display = 'none'; selectedGroupNameEl.textContent = ''; }
            parsersExpanded = false;
            loadParsersAndSettings();
        }
    });

    // Keyboard navigation (Arrow keys, Home, End)
    parsersTabs.addEventListener('keydown', (e) => {
        const keys = ['ArrowRight','ArrowLeft','Home','End'];
        if (!keys.includes(e.key)) return;
        const tabs = Array.from(parsersTabs.querySelectorAll('.tab'));
        const idx = tabs.indexOf(document.activeElement);
        if (e.key === 'ArrowRight') {
            tabs[(idx + 1) % tabs.length].focus();
        } else if (e.key === 'ArrowLeft') {
            tabs[(idx - 1 + tabs.length) % tabs.length].focus();
        } else if (e.key === 'Home') {
            tabs[0].focus();
        } else if (e.key === 'End') {
            tabs[tabs.length - 1].focus();
        }
    });

    // Click on group tag (buttons)
    groupsListEl.addEventListener('click', (e) => {
        const gp = e.target.closest('.group-tag');
        if (!gp) return;
        const name = gp.getAttribute('data-group-name');
        // mark selected visually
        groupsListEl.querySelectorAll('.group-tag').forEach(g => g.classList.remove('group-selected'));
        gp.classList.add('group-selected');

        parsersTabs.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
        const grpTab = parsersTabs.querySelector('[data-tab="groups"]');
        grpTab.classList.add('active'); grpTab.setAttribute('aria-selected','true'); grpTab.tabIndex = 0;

        currentView = { type: 'group', groupName: name };
        if (selectedGroupNameEl) {
            selectedGroupNameEl.style.display = 'block';
            selectedGroupNameEl.textContent = `Группа: ${name}`;
        }
        parsersExpanded = false;
        loadGroupParsers(name);
    });

    // Keyboard support: Enter activates group
    groupsListEl.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
            const gp = document.activeElement.closest('.group-tag');
            if (gp) gp.click();
        }
    });

    // When content script sends updated parsers, do not override chosen group view
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        if (request.action === 'parsersUpdated') {
            if (currentView.type === 'group') {
                // ignore automatic updates while viewing a specific group
                console.log('Skipping parsersUpdated because viewing a group');
                return;
            }
            currentParsers = request.parsers || [];
            displayParsers(currentParsers);
        }
        
        if (request.action === 'settingsUpdated') {
            displayCurrentSettings(request.settings, request.site);
        }

        if (request.action === 'selectorPicked') {
            if (pickedSelectorInput) {
                pickedSelectorInput.value = request.selector || '';
            }
            setPickerState(false);
            if (request.selector) {
                showStatus(serverStatusEl, 'Селектор выбран', 'success');
            } else {
                showStatus(serverStatusEl, 'Не удалось получить селектор', 'warning');
            }
        }

        if (request.action === 'selectorPickerCanceled') {
            setPickerState(false);
            showStatus(serverStatusEl, 'Выбор отменен', 'info');
        }
    });

    // Refresh parsers button
    refreshParsersBtn.addEventListener('click', async () => {
        refreshParsersBtn.innerText = 'Обновление...';
        refreshParsersBtn.disabled = true;
        try {
            if (currentView.type === 'all') {
                await loadAllParsers();
            } else if (currentView.type === 'group' && currentView.groupName) {
                await loadGroupParsers(currentView.groupName);
            } else if (currentView.type === 'groups') {
                await loadGroups();
            } else {
                await loadParsersAndSettings();
            }
            showStatus(serverStatusEl, 'Parsers refreshed', 'success');
        } catch (err) {
            showStatus(serverStatusEl, 'Ошибка обновления', 'error');
        } finally {
            refreshParsersBtn.innerText = 'Обновить';
            refreshParsersBtn.disabled = false;
        }
    });

    // Listen for storage updates to refresh UI
    chrome.storage.onChanged.addListener((changes, area) => {
        if (area === 'local' && changes.disabledParsers) {
            displayParsers(currentParsers);
        }
    });

    // Функция отображения текущих настроек
    function displayCurrentSettings(settings, site) {
        if (!settings) {
            displayDefaultSettings();
            return;
        }

        let html = `
            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Текущий сайт</div>
                    <div class="setting-value">${site || 'Неизвестно'}</div>
                </div>
            </div>

            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Минимальное количество</div>
                    <div class="setting-value">${settings.minDuplicates || 2}</div>
                </div>
                <div class="setting-status ${settings.useMinDuplicates ? 'status-enabled' : 'status-disabled'}">
                    ${settings.useMinDuplicates !== false ? 'Включено' : 'Выключено'}
                </div>
            </div>

            <div class="setting-card">
                <div class="setting-header">
                    <div class="setting-name">Минимальная дата</div>
                    <div class="setting-value">${new Date(settings.minDate || '2025-09-22').toLocaleDateString('ru-RU')}</div>
                </div>
                <div class="setting-status ${settings.useMinDate ? 'status-enabled' : 'status-disabled'}">
                    ${settings.useMinDate !== false ? 'Включено' : 'Выключено'}
                </div>
            </div>
        `;

        if (settings.siteGroups && settings.siteGroups.length > 0) {
            html += `
                <div class="setting-card">
                    <div class="setting-header">
                        <div class="setting-name">Группы сайтов</div>
                        <div class="setting-value">${settings.siteGroups.length}</div>
                    </div>
                    <div class="groups-list">
                        <span class="group-tag current-site">${site}</span>
                        ${settings.siteGroups.map(group => 
                            `<span class="group-tag">${escapeHtml(group)}</span>`
                        ).join('')}
                    </div>
                </div>
            `;
        }

        settingsEl.innerHTML = html;
    }

    // Экранирование HTML
    function escapeHtml(unsafe) {
        if (!unsafe) return '';
        return unsafe.toString()
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    // Сохранение ключа
    saveBtn.addEventListener("click", async () => {
        const key = input.value.trim();
        if (!key) {
            showStatus(keyStatusEl, "Введите API-ключ", "warning");
            return;
        }

        try {
            // Сохраняем ключ локально, затем проверяем его валидность через сервер
            await chrome.storage.local.set({ apiKey: key, my_api_key: key });

            // Запрос к серверу через background (он добавит X-API-Key из storage)
            const res = await apiRequest({ url: API_ENDPOINTS.VALIDATE_KEY, method: 'GET' });
            if (!res || res.status !== 200) {
                // Некорректный ключ — удаляем и показываем ошибку
                try { await chrome.storage.local.remove(['apiKey','my_api_key']); } catch(e) {}
                showStatus(keyStatusEl, "Неверный API-ключ или нет доступа", "error");
                return;
            }

            showStatus(keyStatusEl, "Ключ успешно сохранён и проверен", "success");
            try {
                await loadParsersAndSettings();
                await loadGroups();
            } catch (e) {}
            try {
                const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
                if (tab?.id) {
                    chrome.tabs.sendMessage(tab.id, { action: 'refreshSettingsAndRescan' });
                }
            } catch (e) {}
            // Анимация кнопки
            saveBtn.innerHTML = '<span></span> Успешно';
            setTimeout(() => {
                saveBtn.innerHTML = '<span></span> Сохранить';
            }, 2000);
            
        } catch (err) {
            showStatus(keyStatusEl, "Ошибка при сохранении ключа", "error");
        }
    });

    // Очистка ключа
    clearBtn.addEventListener("click", async () => {
        if (!confirm("Вы уверены, что хотите удалить API-ключ?")) return;
        
        try {
            await chrome.storage.local.remove(["apiKey", "my_api_key"]);
            input.value = "";
            showStatus(keyStatusEl, "Ключ удалён", "success");
            
            // Анимация кнопки
            clearBtn.innerHTML = '<span></span> Удалено';
            setTimeout(() => {
                clearBtn.innerHTML = '<span></span> Очистить';
            }, 2000);
            
        } catch (err) {
            showStatus(keyStatusEl, "Ошибка при удалении ключа", "error");
        }
    });

    // Обновленная функция проверки сервера в обработчике click
checkBtn.addEventListener("click", async () => {
    checkBtn.innerHTML = '<span></span> Проверка...';
    checkBtn.disabled = true;
    
    try {
        // Сначала пробуем fast ping
        let response;
        try {
                response = await apiRequest({ url: API_ENDPOINTS.PING_FAST });
                if (response.status === 200) {
                    const data = response.data;
                    showStatus(serverStatusEl, `Fast ping: ${new Date(data.timestamp).toLocaleTimeString()}`, "success");
                    return;
                }
        } catch (fastError) {
            console.log('Fast ping не доступен...');
        }
        
        // Затем обычный ping
        response = await apiRequest({ url: API_ENDPOINTS.PING });
        if (response.status === 200) {
            showStatus(serverStatusEl, "Сервер доступен", "success");
        } else {
            showStatus(serverStatusEl, `Ошибка сервера (${response.status})`, "warning");
        }
    } catch (err) {
        showStatus(serverStatusEl, "Сервер недоступен", "error");
    } finally {
        checkBtn.innerHTML = '<span></span> Проверить соединение';
        checkBtn.disabled = false;
    }
});

    // Обновление настроек и парсеров
    refreshBtn.addEventListener("click", async () => {
        refreshBtn.innerHTML = '<span></span> Обновление...';
        refreshBtn.disabled = true;
        
        try {
            // Обновляем настройки через content script
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab && tab.id) {
                await chrome.tabs.sendMessage(tab.id, { 
                    action: "refreshSettings" 
                });
            }
            
            // Перезагружаем данные
            await loadParsersAndSettings();
            showStatus(serverStatusEl, "Данные обновлены", "success");
        } catch (error) {
            console.error("Ошибка обновления:", error);
            showStatus(serverStatusEl, "Ошибка обновления", "error");
        } finally {
            refreshBtn.innerHTML = '<span></span> Обновить';
            refreshBtn.disabled = false;
        }
    });

    // Переключатель авто-сканирования
    enabledToggle.addEventListener("change", async () => {
        const enabled = enabledToggle.checked;
        await chrome.storage.local.set({ enabled });
        
        // Уведомляем content script
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab && tab.id) {
                await chrome.tabs.sendMessage(tab.id, { 
                    action: "toggleAutoScan",
                    enabled: enabled
                });
            }
        } catch (error) {
            console.log("Не удалось уведомить content script:", error);
        }
        
        showStatus(serverStatusEl, 
            enabled ? "Авто-сканирование включено" : "Авто-сканирование выключено", 
            "info"
        );
    });

    // Кнопка пересканирования страницы
    rescanBtn.addEventListener('click', async () => {
        rescanBtn.innerHTML = '<span></span> Сканируем...';
        rescanBtn.disabled = true;
        
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab && tab.id) {
                // Отправляем запрос на пересканирование в content script (guarded)
                try {
                    const response = await chrome.tabs.sendMessage(tab.id, { action: "rescanPage" });
                    if (response && response.success) {
                        showStatus(serverStatusEl, "Страница пересканирована", "success");
                        // Обновляем данные после пересканирования
                        setTimeout(async () => { await loadParsersAndSettings(); }, 1000);
                    } else {
                        throw new Error("Content script не ответил");
                    }
                } catch (msgErr) {
                    console.warn('Не удалось отправить сообщение content script:', msgErr);
                    showStatus(serverStatusEl, 'Content script недоступен', 'warning');
                }
            }
        } catch (error) {
            console.error("Ошибка пересканирования:", error);
            showStatus(serverStatusEl, "Ошибка пересканирования", "error");
        } finally {
            rescanBtn.innerHTML = '<span></span> Пересканировать';
            rescanBtn.disabled = false;
        }
    });

    // Автофокус: фокусируемся на поле ключа (если видимо)
    if (input && input.offsetParent !== null) {
        input.focus();
    }

        if (debugEl) debugEl.textContent = 'Инициализация завершена';
    } catch (err) {
        console.error("Popup init error:", err);
        if (debugEl) debugEl.textContent = "Ошибка: " + (err && err.message ? err.message : String(err));
        // Покажем ошибку в UI
        const serverStatusEl = document.getElementById("serverStatus");
        if (serverStatusEl) serverStatusEl.textContent = "Ошибка инициализации: " + (err && err.message ? err.message : String(err));
    }
}

// Запуск инициализации: если DOMContentLoaded ещё не пришёл — повесим слушатель, иначе вызовем сразу
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initPopup);
} else {
    // Document уже загружен — запускаем немедленно
    initPopup();
}