// ==================== КОНФИГУРАЦИЯ ====================
const SERVER_URL = "https://idmanager.site";
const SITE_NAME = window.location.hostname.replace(/^(www\.)/, "");

// КОРРЕКТНЫЕ ЭНДПОИНТЫ ДЛЯ МОДУЛЬНОГО СЕРВЕРА
const API_ENDPOINTS = {
  // Основные эндпоинты для расширения
  GET_ID: (id, site) => `${SERVER_URL}/api/ids/get-id/${encodeURIComponent(id)}?site=${encodeURIComponent(site)}`,
  ADD_ID: `${SERVER_URL}/api/ids/add-id`,
  
  // Парсеры - ПРАВИЛЬНЫЙ ПУТЬ!
  PARSERS: `${SERVER_URL}/api/parsers/parsers`,
  PARSERS_EXPORT: `${SERVER_URL}/api/parsers/parsers/export`,
  
  // Настройки сайта - ПРАВИЛЬНЫЙ ПУТЬ!
  SITE_SETTINGS: `${SERVER_URL}/api/sites/site-settings`,
  
  // Группы сайтов - ПРАВИЛЬНЫЙ ПУТЬ!
  SITE_GROUPS: `${SERVER_URL}/api/sites/site-groups`,
  
  // Пакетная проверка
  BULK_CHECK: `${SERVER_URL}/api/ids/bulk-check-ids`,

    // Заметки
    NOTE_BY_KEY: `${SERVER_URL}/api/ids/note-by-key`,
  
  // Ping
  PING: `${SERVER_URL}/api/ids/ping`,
  PING_FAST: `${SERVER_URL}/api/ids/ping-fast`
};

// Helper: proxy requests through background to attach API key and avoid CORS
function apiRequest({ url, method = 'GET', body = null, headers = {} } = {}) {
    return new Promise((resolve) => {
        chrome.runtime.sendMessage({ action: 'apiRequest', url, method, body, headers }, (resp) => {
            resolve(resp || { status: 0, error: 'no-response' });
        });
    });
}

function showSubscriptionToast(message) {
    try {
        const existing = document.getElementById('idmanager-subscription-toast');
        if (existing) existing.remove();

        const toast = document.createElement('div');
        toast.id = 'idmanager-subscription-toast';
        toast.textContent = message;
        Object.assign(toast.style, {
            position: 'fixed',
            right: '16px',
            bottom: '16px',
            zIndex: '2147483647',
            background: '#fef2f2',
            color: '#991b1b',
            border: '1px solid #fecaca',
            padding: '12px 14px',
            borderRadius: '10px',
            fontSize: '14px',
            maxWidth: '320px',
            boxShadow: '0 10px 30px rgba(0,0,0,0.15)'
        });
        document.body.appendChild(toast);
        setTimeout(() => toast.remove(), 8000);
    } catch (e) {
        // ignore
    }
}

chrome.runtime.onMessage.addListener((request) => {
    if (request?.action !== 'subscriptionExpired') return;
    const err = request?.details?.error || '';
    let message = 'Подписка истекла. Продлите доступ.';
    if (err === 'subscription_grace_expired') {
        message = 'Подписка истекла более 7 дней назад. Доступ закрыт.';
    } else if (err === 'subscription_expired_admin_add_disabled') {
        message = 'Подписка истекла. Добавление ID недоступно до продления.';
    }
    showSubscriptionToast(message);
});

chrome.runtime.onMessage.addListener((request) => {
    if (request?.action !== 'clearSession') return;
    try {
        // Clear caches
        siteParsers = [];
        lastParsersLoad = 0;
        siteSettings = {
            minDuplicates: 2,
            minDate: "2025-09-22",
            useMinDuplicates: true,
            useMinDate: true,
            siteGroups: []
        };
        DYNAMIC_SELECTORS = '';
        recordsCache.clear();
        updateInProgress.clear();
        idToElements.clear();
        pendingCheckIds.clear();
        pendingAddIds.clear();
        pendingAddResolvers.clear();
        pendingNodes.clear();

        // Remove UI controls
        document.querySelectorAll('.universal-id-controls').forEach(el => el.remove());
        document.querySelectorAll('.universal-id-flagged').forEach(el => el.classList.remove('universal-id-flagged'));
    } catch (e) {
        console.warn('clearSession handler error', e);
    }
});

// ==================== ГЛОБАЛЬНЫЕ ПЕРЕМЕННЫЕ ====================
// ДИНАМИЧЕСКИЕ СЕЛЕКТОРЫ
let DYNAMIC_SELECTORS = '';

// Кэш настроек сайта и групп
let siteSettings = {
    minDuplicates: 2,
    minDate: "2025-09-22",
    useMinDuplicates: true,
    useMinDate: true,
    siteGroups: []
};

// Кэш парсеров
let siteParsers = [];
let lastParsersLoad = 0;
// Кешируем парсеры дольше, чтобы снизить частоту запросов к серверу
const PARSERS_CACHE_TIME = 5 * 60 * 1000; // 5 минут

// КЭШ ДЛЯ ЗАПИСЕЙ И СОСТОЯНИЕ
const recordsCache = new Map();
const updateInProgress = new Set();
let lastSettingsLoad = 0;  // <-- ДОБАВЬТЕ ЭТУ ПЕРЕМЕННУЮ!
// Настройки тоже кешируем дольше (можно уменьшить при необходимости)
const SETTINGS_CACHE_TIME = 2 * 60 * 1000; // 2 минуты

// URL для отслеживания SPA навигации
let currentUrl = window.location.href;

// Наблюдатель
let observer = null;

// ==================== КЭШИ И ОЧЕРЕДИ ДЛЯ БЫСТРОДЕЙСТВИЯ ====================
// Кэш API ключа (чтобы не дергать storage каждый раз)
let cachedApiKey = null;
let apiKeyLoaded = false;

// Карта ID -> элементы для точечного обновления
const idToElements = new Map();
const elementToId = new WeakMap();
const elementToAnchor = new WeakMap();
const nicknameCacheByUrl = new Map();
const nicknameInFlight = new Map();
let selectorPickerActive = false;
let selectorPickerLastHover = null;
let selectorPickerHint = null;

// Очередь на пакетную проверку ID
const pendingCheckIds = new Set();
let pendingCheckTimer = null;
const CHECK_DEBOUNCE_MS = 200;

// Очередь на пакетную отправку ID
const pendingAddIds = new Set();
let pendingAddTimer = null;
let pendingAddApiKey = null;
const ADD_DEBOUNCE_MS = 250;
const pendingAddResolvers = new Map();

// Очередь на обработку новых DOM-узлов
const pendingNodes = new Set();
let pendingNodesTimer = null;
const NODES_DEBOUNCE_MS = 120;

// ==================== ИСПРАВЛЕННЫЕ ФУНКЦИИ ====================

// Загрузка парсеров - ИСПРАВЛЕННЫЙ ПУТЬ
// Загрузка парсеров - ОПТИМИЗИРОВАННЫЙ ПУТЬ
async function loadSiteParsers() {
    const now = Date.now();
    if (now - lastParsersLoad < PARSERS_CACHE_TIME) {
        return siteParsers;
    }
    
    try {
        // console.log(`Загрузка парсеров для: ${SITE_NAME}`);
        
        // Приоритет у нового эндпоинта экспорта
        let parsers = [];
        
        try {
            const res = await apiRequest({ url: `${API_ENDPOINTS.PARSERS_EXPORT}?site=${encodeURIComponent(SITE_NAME)}` });
            if (res.status === 200) {
                const data = res.data;
                if (data && data.success && data.parsers) {
                    parsers = data.parsers.filter(p => p.enabled !== false);
                }
            }
        } catch (exportError) {
            // fallback
        }
        
        // Fallback если экспорт не сработал
        if (parsers.length === 0) {
            const res = await apiRequest({ url: `${API_ENDPOINTS.PARSERS}?site=${encodeURIComponent(SITE_NAME)}` });
            if (res.status === 200) {
                const serverParsers = res.data;
                parsers = Array.isArray(serverParsers) ? serverParsers.filter(p => p.is_active !== false) : (serverParsers.parsers || []);
            }
        }
        
        // Apply local disabled parsers filter (popup-controlled feature)
        const _storage = await new Promise(r => chrome.storage.local.get(['disabledParsers'], r));
        const _disabledSet = new Set((_storage.disabledParsers || []).map(x => Number(x)));
        siteParsers = parsers.filter(p => !_disabledSet.has(Number(p.id)));
        lastParsersLoad = now;
        // Если сайт принадлежит группам, попробуем подтянуть парсеры, привязанные к этим группам
        try {
            const groupNames = siteSettings.siteGroups || [];
            if (Array.isArray(groupNames) && groupNames.length > 0) {
                // Собираме список id парсеров для всех групп
                const parserIdSet = new Set();
                for (const groupName of groupNames) {
                    try {
                        try {
                            const gpRes = await apiRequest({ url: `${SERVER_URL}/api/sites/site-groups/${encodeURIComponent(groupName)}/parsers` });
                            if (gpRes.status === 200) {
                                const gpData = gpRes.data;
                                const list = gpData && (Array.isArray(gpData.parsers) ? gpData.parsers : gpData) || [];
                                list.forEach(pid => parserIdSet.add(Number(pid)));
                            }
                        } catch (e) {
                            console.warn('Не удалось получить парсеры группы', groupName, e);
                        }
                    } catch (e) {
                        console.warn('Не удалось получить парсеры группы', groupName, e);
                    }
                }

                if (parserIdSet.size > 0) {
                    try {
                        // Получаем все парсеры и добавляем нужные по id
                        const allRes = await apiRequest({ url: `${API_ENDPOINTS.PARSERS}` });
                        if (allRes.status === 200) {
                            const allParsersData = allRes.data;
                            const allList = Array.isArray(allParsersData) ? allParsersData : (allParsersData.parsers || []);
                            const extra = allList.filter(p => parserIdSet.has(Number(p.id)));
                            // Объединяем и убираем дубликаты по id
                            const merged = [...siteParsers];
                            const existingIds = new Set(merged.map(p => Number(p.id)));
                            extra.forEach(p => { if (!existingIds.has(Number(p.id))) merged.push(p); });
                            siteParsers = merged.filter(p => !_disabledSet.has(Number(p.id)));
                        }
                    } catch (e) {
                        console.warn('Ошибка при загрузке дополнительных парсеров по группам:', e);
                    }
                }
            }
        } catch (e) {
            console.warn('Ошибка при обработке парсеров групп:', e);
        }

        updateDynamicSelectors();
        
        // Уведомляем popup
        chrome.runtime.sendMessage({
            action: 'parsersUpdated',
            parsers: siteParsers,
            parsersCount: siteParsers.length,
            selectorsCount: Array.from(new Set(siteParsers.flatMap(p => p.selectors || []))).length,
            site: SITE_NAME,
            groups: siteSettings.siteGroups || []
        });
        
        return siteParsers;
        
    } catch (e) {
        console.warn("Ошибка загрузки парсеров:", e);
        siteParsers = [];
        // Notify popup/background that parsers failed to load so UI can update
        try {
            chrome.runtime.sendMessage({
                action: 'parsersUpdated',
                parsers: [],
                parsersCount: 0,
                selectorsCount: 0,
                site: SITE_NAME,
                error: String(e && e.message ? e.message : e)
            });
        } catch (msgErr) {
            console.warn('Failed to notify parsersUpdated on error:', msgErr);
        }
        return [];
    }
}

// Allow popup to request immediate parser refresh
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'refreshParsersFromPopup') {
        lastParsersLoad = 0;
        loadSiteParsers().then(() => {
            // After reloading parsers, re-scan page
            try { scanPageWithParsers(); } catch (err) { /* ignore */ }
            sendResponse({ success: true });
        }).catch(err => {
            console.warn('Ошибка при refreshParsersFromPopup:', err);
            sendResponse({ success: false });
        });
        return true; // Will respond asynchronously
    }
});

// React to local disabledParsers updates
chrome.storage.onChanged.addListener((changes, area) => {
    if (area === 'local' && changes.disabledParsers) {
        lastParsersLoad = 0;
        loadSiteParsers();
    }
    if (area === 'local' && (changes.apiKey || changes.my_api_key)) {
        const newKey = (changes.apiKey && changes.apiKey.newValue) || (changes.my_api_key && changes.my_api_key.newValue) || null;
        cachedApiKey = newKey;
        apiKeyLoaded = true;
    }
});


// Загрузка настроек сайта - ИСПРАВЛЕННЫЙ ПУТЬ
async function loadSiteSettings() {
    const now = Date.now();
    if (now - lastSettingsLoad < SETTINGS_CACHE_TIME) {
        return;
    }
    
    try {
        // ИСПРАВЛЕННЫЙ ПУТЬ!
        const res = await apiRequest({ url: `${API_ENDPOINTS.SITE_SETTINGS}?site=${encodeURIComponent(SITE_NAME)}` });

        if (res.status === 200) {
            const settings = res.data;
            
            siteSettings = {
                minDuplicates: settings.minDuplicates == null ? 2 : Number(settings.minDuplicates),
                minDate: settings.minDate || getDefaultMinDate(),
                useMinDuplicates: settings.useMinDuplicates === false ? false : true,
                useMinDate: settings.useMinDate === false ? false : true,
                siteGroups: settings.siteGroups || []
            };
            
            chrome.storage.local.set({ currentSiteSettings: siteSettings });
            lastSettingsLoad = now;
            
            // Уведомляем popup об обновлении
            chrome.runtime.sendMessage({
                action: 'settingsUpdated',
                settings: siteSettings,
                site: SITE_NAME
            });
        } else {
            console.warn(`Ошибка загрузки настроек: HTTP ${res.status}`);
            // Устанавливаем настройки по умолчанию
            siteSettings = {
                minDuplicates: 2,
                minDate: getDefaultMinDate(),
                useMinDuplicates: true,
                useMinDate: true,
                siteGroups: []
            };
        }
    } catch (e) {
        console.warn("Ошибка загрузки настроек:", e);
        // Устанавливаем настройки по умолчанию при ошибке
        siteSettings = {
            minDuplicates: 2,
            minDate: getDefaultMinDate(),
            useMinDuplicates: true,
            useMinDate: true,
            siteGroups: []
        };
        // Notify popup/background about settings load failure
        try {
            chrome.runtime.sendMessage({
                action: 'settingsUpdated',
                settings: siteSettings,
                site: SITE_NAME,
                error: String(e && e.message ? e.message : e)
            });
        } catch (msgErr) {
            console.warn('Failed to notify settingsUpdated on error:', msgErr);
        }
    }
}

function getDefaultMinDate() {
    const date = new Date();
    date.setDate(date.getDate() - 30);
    return date.toISOString().split('T')[0];
}

// ==================== ИСПРАВЛЕННАЯ ФУНКЦИЯ ДЛЯ РАБОТЫ С ПУСТЫМИ СЕЛЕКТОРАМИ ====================

// Функция для получения текущих селекторов (с защитой)
function getCurrentSelectors() {
    if (!DYNAMIC_SELECTORS || DYNAMIC_SELECTORS.trim() === '') {
        // Возвращаем безопасный селектор, который ничего не найдет
        return '.universal-id-no-elements-found';
    }
    return DYNAMIC_SELECTORS;
}

// ==================== УТИЛИТЫ ПРОИЗВОДИТЕЛЬНОСТИ ====================

function ensureHighlightStyle() {
    if (document.getElementById('universal-id-flag-style')) return;
    const st = document.createElement('style');
    st.id = 'universal-id-flag-style';
    st.textContent = `
        .universal-id-flagged {
            background-color: rgba(255,180,180,0.35) !important;
            outline: 2px solid #ff3366 !important;
            outline-offset: 1px !important;
            box-shadow: 0 0 10px rgba(255, 51, 102, 0.5) !important;
            transition: background-color 0.2s ease;
        }
    `;
    document.head.appendChild(st);
}

function ensurePickerStyle() {
    if (document.getElementById('universal-id-picker-style')) return;
    const st = document.createElement('style');
    st.id = 'universal-id-picker-style';
    st.textContent = `
        .universal-id-picker-hover {
            outline: 2px solid #3b82f6 !important;
            outline-offset: 1px !important;
            box-shadow: 0 0 0 2px rgba(59, 130, 246, 0.2) !important;
            background-color: rgba(59, 130, 246, 0.08) !important;
        }
        #universal-id-picker-hint {
            position: fixed;
            top: 14px;
            right: 14px;
            z-index: 2147483647;
            background: #111827;
            color: #f9fafb;
            padding: 8px 10px;
            border-radius: 8px;
            font-size: 12px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            box-shadow: 0 8px 18px rgba(0,0,0,0.25);
        }
    `;
    document.head.appendChild(st);
}

function cssEscapeValue(value) {
    if (window.CSS && typeof window.CSS.escape === 'function') {
        return window.CSS.escape(value);
    }
    return String(value).replace(/[^a-zA-Z0-9_-]/g, '\\$&');
}

function buildCssSelector(el) {
    if (!el || el.nodeType !== 1) return '';
    if (el.id) return `#${cssEscapeValue(el.id)}`;

    const parts = [];
    let node = el;
    while (node && node.nodeType === 1 && parts.length < 5) {
        let part = node.tagName.toLowerCase();
        const classList = Array.from(node.classList || []).filter(c => c && !c.startsWith('universal-id-'));
        if (classList.length > 0) {
            part += '.' + classList.slice(0, 3).map(cssEscapeValue).join('.');
        }
        const parent = node.parentElement;
        if (parent) {
            const siblings = Array.from(parent.children).filter(ch => ch.tagName === node.tagName);
            if (siblings.length > 1) {
                const idx = siblings.indexOf(node) + 1;
                part += `:nth-of-type(${idx})`;
            }
        }
        parts.unshift(part);
        if (parent && parent.id) {
            parts.unshift(`#${cssEscapeValue(parent.id)}`);
            break;
        }
        node = parent;
    }
    return parts.join(' > ');
}

function isPickerIgnored(el) {
    if (!el || !el.closest) return false;
    if (el.id === 'universal-id-picker-hint') return true;
    if (el.closest('.universal-id-controls')) return true;
    return false;
}

function stopSelectorPicker(sendCancel = true) {
    if (!selectorPickerActive) return;
    selectorPickerActive = false;

    if (selectorPickerLastHover) {
        selectorPickerLastHover.classList.remove('universal-id-picker-hover');
        selectorPickerLastHover = null;
    }

    if (selectorPickerHint && selectorPickerHint.parentNode) {
        selectorPickerHint.parentNode.removeChild(selectorPickerHint);
    }
    selectorPickerHint = null;

    document.removeEventListener('mousemove', selectorPickerOnMove, true);
    document.removeEventListener('click', selectorPickerOnClick, true);
    document.removeEventListener('keydown', selectorPickerOnKeyDown, true);

    if (sendCancel) {
        try {
            chrome.runtime.sendMessage({ action: 'selectorPickerCanceled' });
        } catch (e) {}
    }
}

function selectorPickerOnMove(e) {
    if (!selectorPickerActive) return;
    const target = e.target;
    if (!target || target.nodeType !== 1 || isPickerIgnored(target)) return;
    if (selectorPickerLastHover && selectorPickerLastHover !== target) {
        selectorPickerLastHover.classList.remove('universal-id-picker-hover');
    }
    selectorPickerLastHover = target;
    selectorPickerLastHover.classList.add('universal-id-picker-hover');
}

function selectorPickerOnClick(e) {
    if (!selectorPickerActive) return;
    e.preventDefault();
    e.stopPropagation();
    const target = e.target;
    if (!target || target.nodeType !== 1 || isPickerIgnored(target)) return;
    const selector = buildCssSelector(target);
    stopSelectorPicker(false);
    try {
        chrome.runtime.sendMessage({ action: 'selectorPicked', selector, tag: target.tagName.toLowerCase() });
    } catch (err) {}
}

function selectorPickerOnKeyDown(e) {
    if (!selectorPickerActive) return;
    if (e.key === 'Escape') {
        e.preventDefault();
        stopSelectorPicker(true);
    }
}

function startSelectorPicker() {
    if (selectorPickerActive) return;
    selectorPickerActive = true;
    ensurePickerStyle();

    selectorPickerHint = document.createElement('div');
    selectorPickerHint.id = 'universal-id-picker-hint';
    selectorPickerHint.textContent = 'Выберите элемент. Esc - отмена';
    document.body.appendChild(selectorPickerHint);

    document.addEventListener('mousemove', selectorPickerOnMove, true);
    document.addEventListener('click', selectorPickerOnClick, true);
    document.addEventListener('keydown', selectorPickerOnKeyDown, true);
}

function registerElementId(el, id) {
    if (!el || !id) return;
    const prevId = elementToId.get(el);
    if (prevId && prevId !== id) {
        const prevSet = idToElements.get(prevId);
        if (prevSet) {
            prevSet.delete(el);
            if (prevSet.size === 0) idToElements.delete(prevId);
        }
    }
    elementToId.set(el, id);
    if (!idToElements.has(id)) idToElements.set(id, new Set());
    idToElements.get(id).add(el);
}

function getRegisteredIdForElement(el) {
    return elementToId.get(el) || null;
}

function getAnchorForElement(el) {
    return elementToAnchor.get(el) || el;
}

function cleanupDeadElements(id) {
    const set = idToElements.get(id);
    if (!set) return;
    for (const el of Array.from(set)) {
        if (!el || !document.contains(el)) {
            set.delete(el);
        }
    }
    if (set.size === 0) idToElements.delete(id);
}

function applyBulkResults(results, ids) {
    const now = Date.now();
    const keys = Array.isArray(ids) ? ids : Object.keys(results || {});
    for (const rawId of keys) {
        const id = String(rawId);
        const resForId = results[id] || results[rawId] || results[id.toString()] || [];
        const normalized = Array.isArray(resForId.records)
            ? resForId.records
            : (Array.isArray(resForId) ? resForId : []);

        recordsCache.set(id, { records: normalized, timestamp: now });

        cleanupDeadElements(id);
        const elements = idToElements.get(id);
        if (elements && elements.size > 0) {
            for (const el of Array.from(elements)) {
                const anchor = getAnchorForElement(el);
                const btn = anchor.querySelector('.universal-upload-btn');
                if (btn) updateUploadButton(btn, normalized, id);
                applyHighlight(anchor, shouldHighlight(normalized));
            }
        }
    }
}

function scheduleBulkCheck() {
    if (pendingCheckTimer) return;
    pendingCheckTimer = setTimeout(async () => {
        pendingCheckTimer = null;
        const ids = Array.from(pendingCheckIds);
        pendingCheckIds.clear();
        if (ids.length === 0) return;
        try {
            const results = await bulkCheckIds(ids);
            applyBulkResults(results, ids);
        } catch (e) {
            console.warn('Ошибка в пакетной проверке IDs:', e);
        }
    }, CHECK_DEBOUNCE_MS);
}

function enqueueCheckId(id) {
    if (!id) return;
    pendingCheckIds.add(String(id));
    scheduleBulkCheck();
}

function enqueueAddId(id, apiKey) {
    if (!id || !apiKey) return Promise.resolve(false);
    pendingAddIds.add(String(id));
    pendingAddApiKey = apiKey;
    return new Promise((resolve) => {
        if (!pendingAddResolvers.has(String(id))) pendingAddResolvers.set(String(id), []);
        pendingAddResolvers.get(String(id)).push(resolve);
        if (!pendingAddTimer) {
            pendingAddTimer = setTimeout(async () => {
                pendingAddTimer = null;
                const ids = Array.from(pendingAddIds);
                pendingAddIds.clear();
                const apiKeyToUse = pendingAddApiKey;
                pendingAddApiKey = null;
                let success = false;
                try {
                    success = await bulkAddIds(ids, apiKeyToUse);
                } catch (e) {
                    console.warn('Ошибка в пакетной отправке IDs:', e);
                    success = false;
                }
                for (const sentId of ids) {
                    const resolvers = pendingAddResolvers.get(String(sentId)) || [];
                    pendingAddResolvers.delete(String(sentId));
                    resolvers.forEach(r => r(success));
                }
            }, ADD_DEBOUNCE_MS);
        }
    });
}

function queueNodesForEnhance(nodes, currentSelectors) {
    if (!currentSelectors || currentSelectors === '.universal-id-no-elements-found') return;
    for (const node of nodes) {
        if (!node || node.nodeType !== 1) continue;
        try {
            if (node.matches && node.matches(currentSelectors)) {
                pendingNodes.add(node);
            }
            if (node.querySelectorAll) {
                node.querySelectorAll(currentSelectors).forEach(el => pendingNodes.add(el));
            }
        } catch (e) {
            // игнорируем ошибки селекторов
        }
    }

    if (pendingNodesTimer) return;
    pendingNodesTimer = setTimeout(() => {
        pendingNodesTimer = null;
        const batch = Array.from(pendingNodes);
        pendingNodes.clear();
        batch.forEach(el => {
            if (!el._universalEnhanced) enhanceCard(el);
        });
    }, NODES_DEBOUNCE_MS);
}
// ==================== ФУНКЦИИ СЕЛЕКТОРОВ ====================

// Функция для обновления селекторов из парсеров
function updateDynamicSelectors() {
    if (!siteParsers || siteParsers.length === 0) {
        console.log('Нет парсеров для обновления селекторов');
        DYNAMIC_SELECTORS = ''; // Очищаем селекторы если нет парсеров
        return;
    }
    
    const allSelectors = new Set();
    
    // ДОБАВЛЯЕМ ТОЛЬКО СЕЛЕКТОРЫ ИЗ ПАРСЕРОВ - без базовых
    siteParsers.forEach(parser => {
        if (parser.selectors && Array.isArray(parser.selectors)) {
            parser.selectors.forEach(selector => {
                if (selector && selector.trim() && selector.length > 2) {
                    allSelectors.add(selector.trim());
                }
            });
        }
    });
    
    // Обновляем глобальную переменную
    DYNAMIC_SELECTORS = Array.from(allSelectors).join(', ');
    
    // console.log(`Обновлены динамические селекторы из парсеров: ${allSelectors.size} селекторов`, Array.from(allSelectors));
    
    // Уведомляем об обновлении
    chrome.runtime.sendMessage({
        action: 'selectorsUpdated',
        selectorsCount: allSelectors.size,
        selectors: Array.from(allSelectors).slice(0, 10)
    });
}

// Функция проверки соответствия элемента селекторам парсеров
function elementMatchesAnyParserSelector(element) {
    if (!siteParsers || siteParsers.length === 0) {
        return true; // Если парсеров нет, пропускаем все элементы
    }

    return Boolean(getMatchingParserForElement(element));
}

function getMatchingParserForElement(element) {
    if (!siteParsers || siteParsers.length === 0 || !element) return null;

    for (const parser of siteParsers) {
        if (parser.selectors && Array.isArray(parser.selectors)) {
            for (const selector of parser.selectors) {
                try {
                    if (selector && element.matches && element.matches(selector)) {
                        return parser;
                    }
                } catch (selectorError) {
                    console.warn(`Ошибка в селекторе "${selector}" парсера "${parser.name}":`, selectorError);
                }
            }
        }
    }

    return null;
}

function resolveButtonAnchor(element, parser) {
    if (!element) return null;
    const rawSelector = parser && (parser.buttonAnchorSelector || parser.button_anchor_selector);
    const selector = rawSelector && String(rawSelector).trim();
    if (!selector) return element;

    try {
        if (element.matches && element.matches(selector)) return element;
    } catch (e) {
        // ignore invalid selector
    }

    try {
        if (element.closest) {
            const closestMatch = element.closest(selector);
            if (closestMatch) return closestMatch;
        }
    } catch (e) {
        // ignore invalid selector
    }

    try {
        const found = element.querySelector(selector);
        if (found) return found;
    } catch (e) {
        // ignore invalid selector
    }

    return element;
}

function shouldUseNickname(parser) {
    if (!parser) return false;
    return parser.useNickname === true || parser.use_nickname === true || !!(parser.nicknameSelector || parser.nickname_selector);
}

function getDetailUrlForElement(element, parser) {
    if (!element) return null;
    const rawSelector = parser && (parser.detailLinkSelector || parser.detail_link_selector);
    const selector = rawSelector && String(rawSelector).trim();
    let linkEl = null;

    if (selector) {
        try {
            if (element.matches && element.matches(selector)) {
                linkEl = element;
            } else {
                linkEl = element.querySelector(selector);
            }
        } catch (e) {
            // ignore invalid selector
        }
    }

    if (!linkEl) {
        linkEl = element.querySelector('a[href]');
    }

    if (!linkEl) return null;
    const href = linkEl.getAttribute('href') || '';
    if (!href) return null;
    try {
        return new URL(href, window.location.href).toString();
    } catch (e) {
        return null;
    }
}

function extractNicknameFromHtml(html, parser) {
    if (!html) return null;

    const rawSelector = parser && (parser.nicknameSelector || parser.nickname_selector);
    const selector = rawSelector && String(rawSelector).trim();

    if (selector) {
        try {
            const doc = new DOMParser().parseFromString(html, 'text/html');
            const el = doc.querySelector(selector);
            const text = el ? String(el.textContent || '').trim() : '';
            if (text) return text;
        } catch (e) {
            // ignore selector errors
        }
    }

    const m = html.match(/\/uzivatel-\d+\/([^/\"']+)\.aspx/i);
    if (m && m[1]) return m[1];
    return null;
}

async function resolveNicknameFromDetailUrl(url, parser) {
    if (!url) return null;
    if (nicknameCacheByUrl.has(url)) return nicknameCacheByUrl.get(url);
    if (nicknameInFlight.has(url)) return nicknameInFlight.get(url);

    const p = (async () => {
        try {
            const res = await fetch(url, { credentials: 'include' });
            if (!res.ok) return null;
            const html = await res.text();
            const nickname = extractNicknameFromHtml(html, parser);
            if (nickname) nicknameCacheByUrl.set(url, nickname);
            return nickname || null;
        } catch (e) {
            return null;
        } finally {
            nicknameInFlight.delete(url);
        }
    })();

    nicknameInFlight.set(url, p);
    return p;
}

async function resolveNicknameForElement(element, parser) {
    const url = getDetailUrlForElement(element, parser);
    if (!url) return null;
    return resolveNicknameFromDetailUrl(url, parser);
}

// ==================== ПАРСЕРЫ ID ====================

function extractIDFromElement(el) {
    if (!el || !el.nodeType) {
        return null;
    }
    
    // ИСПОЛЬЗУЕМ ТОЛЬКО ПАРСЕРЫ - никаких стандартных методов
    if (siteParsers.length > 0) {
        const id = extractIDWithParsers(el);
        if (id && isValidId(id)) {
            // console.log(`Парсер извлёк ВАЛИДНЫЙ ID: ${id}`);
            return id;
        }
    }
    
    return null;
}

// Улучшенная функция парсеров
function extractIDWithParsers(element) {
    const html = element.outerHTML;
    let foundId = null;
    
    if (!siteParsers || siteParsers.length === 0) {
        return null;
    }
    
    for (const parser of siteParsers) {
        if (foundId) break;
        
        try {
            // console.log(`Применяем парсер "${parser.name}" для сайта ${parser.site}`);

            if (shouldUseNickname(parser) && (parser.nicknameSelector || parser.nickname_selector)) {
                const rawSelector = parser.nicknameSelector || parser.nickname_selector;
                const selector = rawSelector && String(rawSelector).trim();
                if (selector) {
                    try {
                        let nickEl = null;
                        if (element.matches && element.matches(selector)) {
                            nickEl = element;
                        } else {
                            nickEl = element.querySelector(selector);
                        }
                        if (nickEl) {
                            const text = String(nickEl.textContent || '').trim();
                            const token = text.split(/[\s,]/).filter(Boolean)[0] || '';
                            if (token) {
                                foundId = token;
                                break;
                            }
                        }
                    } catch (selectorError) {
                        console.warn(`Ошибка nickname selector в парсере "${parser.name}":`, selectorError.message);
                    }
                }
            }

            if (parser.pattern && parser.pattern.length > 2) {
                try {
                    const regex = new RegExp(parser.pattern, 'gi');
                    regex.lastIndex = 0; // Сбрасываем позицию
                    
                    const match = regex.exec(html);
                    if (match) {
                        // Используем первую группу захвата или всё совпадение
                        foundId = match[1] || match[0];
                        if (foundId) {
                            foundId = foundId.trim();
                            // console.log(`Парсер "${parser.name}" извлёк: ${foundId}`);
                            break;
                        }
                    }
                } catch (regexError) {
                    console.warn(`Ошибка regex в парсере "${parser.name}":`, regexError.message);
                    continue;
                }
            }
        } catch (error) {
            console.error(`Общая ошибка в парсере "${parser.name}":`, error);
        }
    }
    
    return foundId;
}

// Функция проверки валидности ID
function isValidId(id) {
    if (!id || typeof id !== 'string') return false;
    
    const cleanId = id.trim();
    
    // Сначала пропустим явно мусорные значения
    const invalidPatterns = [
        /^factfile$/i,
        /^testresult$/i, 
        /^partnersuggestions$/i,
        /^matches$/i,
        /^profile$/i,
        /^partner$/i,
        /^https?:\/\//,
        /^\/\//,
        /^[.#]/,
        /^[\s\S]{1,2}$/ // Очень короткие (1-2 символа)
    ];
    
    for (const pattern of invalidPatterns) {
        if (pattern && pattern.test(cleanId)) {
            console.log(`Отсеян мусорный ID: ${cleanId}`);
            return false;
        }
    }
    
    // НОВЫЕ ПРАВИЛА ВАЛИДАЦИИ - более гибкие!
    const validPatterns = [
        /^[A-Z0-9]{8,12}$/, // Parship ID (PSPYD8M2)
        /^[a-zA-Z0-9_-]{4,30}$/, // Общие alphanumeric ID - ИЗМЕНИЛИ с 6 на 4!
        /^\d{4,20}$/, // Числовые ID
        /^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/i, // UUID
        /^[a-zA-Z0-9]{4,35}$/, // Длинные ID - ИЗМЕНИЛИ с 15 на 4!
        /^[a-z0-9]{3,10}$/i, // Короткие буквенно-цифровые (типа x0gib, squcy)
        /^[a-z]{4,10}$/i, // Чисто буквенные 4-10 символов
        /^[A-Za-z0-9_-]{20,120}$/, // Длинные токены (Badoo data-qa-user-id и похожие)
        /^[\p{L}0-9_.-]{3,40}$/u // Unicode ники (например Exrügi77, roman.w)
    ];
    
    // Проверяем соответствие
    const isValid = validPatterns.some(pattern => pattern.test(cleanId));
    
    // ДОПОЛНИТЕЛЬНО: проверка на наличие хоть одного числа или буквы
    const hasLetter = /[\p{L}]/u.test(cleanId);
    const hasNumber = /\d/.test(cleanId);
    const hasSpecial = /[_.-]/.test(cleanId);
    
    const hasValidChars = hasLetter || hasNumber || hasSpecial;
    
    const finalIsValid = isValid && hasValidChars && cleanId.length >= 3;
    
    // console.log(`Проверка ID "${cleanId}":`, {
    //     length: cleanId.length,
    //     hasLetter,
    //     hasNumber,
    //     hasSpecial,
    //     matchesPattern: isValid,
    //     final: finalIsValid ? 'VALID' : 'INVALID'
    // });
    
    return finalIsValid;
}

// ==================== ПЕРЕСКАНИРОВАНИЕ СТРАНИЦЫ ====================

// Функция пересканирования страницы с новыми селекторами - ТОЛЬКО ДЛЯ НОВЫХ ЭЛЕМЕНТОВ
function rescanPageWithNewSelectors() {
    // console.log('Пересканирование страницы с обновленными селекторами');
    
    // НЕ удаляем старые контролы! Только сбрасываем флаги для НОВЫХ элементов
    const currentSelectors = getCurrentSelectors();
    const allElements = document.querySelectorAll(currentSelectors);
    
    // console.log(`Найдено элементов по новым селекторам: ${allElements.length}`);
    
    // Обрабатываем только элементы без флага enhanced
    allElements.forEach(el => {
        if (!el._universalEnhanced) {
            enhanceCard(el);
        }
    });
}

// "Тихое" обновление статуса без пересоздания контролов
async function quietUpdateCardStatuses() {
    const currentSelectors = getCurrentSelectors();
    const cards = document.querySelectorAll(currentSelectors);
    
    // console.log(`Тихое обновление статусов: ${cards.length} карточек`);
    
    for (const el of cards) {
        if (el._universalEnhanced) {
            const id = extractIDFromElement(el);
            if (id) {
                await updateCardStatus(id, el);
            }
        }
    }
}


// ПАССИВНОЕ сканирование - только поиск без отправки
function scanPageWithParsers() {
    if (siteParsers.length === 0) {
        // console.log("Нет активных парсеров для сканирования");
        return;
    }
    
    try {
        const pageHtml = document.documentElement.outerHTML;
        const foundIds = new Set();
        
        // console.log(`Сканирование страницы с ${siteParsers.length} парсерами (только поиск)`);
        
        // Применяем парсеры для поиска ID по всей странице
        for (const parser of siteParsers) {
            try {
                if (parser.pattern && parser.pattern.length > 2) {
                    try {
                        const regex = new RegExp(parser.pattern, 'gi');
                        let match;
                        regex.lastIndex = 0;
                        
                        while ((match = regex.exec(pageHtml)) !== null) {
                            let extractedId = match[1] || match[0];
                            
                            if (extractedId && extractedId.length >= 3 && isValidId(extractedId)) {
                                extractedId = extractedId.trim();
                                // console.log(`Сканирование: парсер "${parser.name}" нашёл ID: ${extractedId}`);
                                foundIds.add(extractedId);
                            }
                        }
                    } catch (regexError) {
                        console.warn(`Ошибка regex в сканировании (${parser.name}):`, regexError);
                    }
                }
            } catch (error) {
                console.error(`Ошибка в парсере ${parser.name} при сканировании:`, error);
            }
        }
        
        // ТОЛЬКО ЛОГИРОВАНИЕ - НИКАКОЙ ОТПРАВКИ
        if (foundIds.size > 0) {
            // console.log(`Сканирование найдено ${foundIds.size} ID (не отправляются автоматически):`, Array.from(foundIds));
            
            // Можно показать уведомление о найденных ID, но не отправлять
            showNotification(`Найдено ${foundIds.size} ID на странице`);
        }
        
    } catch (error) {
        console.error('Ошибка сканирования:', error);
    }
}

// ==================== ЛОГИКА ПОДСВЕТКИ ====================

function shouldHighlight(records) {
    if (!Array.isArray(records) || records.length === 0) {
        // console.log('[shouldHighlight] Нет записей для подсветки');
        return false;
    }

    const count = records.length;
    const useDup = siteSettings.useMinDuplicates;
    const useDate = siteSettings.useMinDate;

    // console.log('[shouldHighlight] siteSettings:', siteSettings);
    // console.log('[shouldHighlight] records:', records);

    if (!useDup && !useDate) {
        // console.log('[shouldHighlight] Подсветка всегда (оба выключены)');
        return true;
    }

    let meetsDuplicates = false;
    let meetsDate = false;

    if (useDup) {
        meetsDuplicates = count >= siteSettings.minDuplicates;
        // console.log(`[shouldHighlight] meetsDuplicates: ${meetsDuplicates} (count=${count}, minDuplicates=${siteSettings.minDuplicates})`);
    }

    if (useDate) {
        const thresholdDate = new Date(siteSettings.minDate);
        meetsDate = records.some(r => {
            if (!r.created_at) return false;
            const recordDate = new Date(r.created_at);
            const valid = !isNaN(recordDate) && recordDate >= thresholdDate;
            // console.log(`[shouldHighlight] Проверка даты: created_at=${r.created_at}, recordDate=${recordDate}, thresholdDate=${thresholdDate}, valid=${valid}`);
            return valid;
        });
        // console.log(`[shouldHighlight] meetsDate: ${meetsDate} (minDate=${siteSettings.minDate})`);
    }

    if (useDup && !useDate) {
        // console.log(`[shouldHighlight] Итог meetsDuplicates: ${meetsDuplicates}`);
        return meetsDuplicates;
    } else if (!useDup && useDate) {
        // console.log(`[shouldHighlight] Итог meetsDate: ${meetsDate}`);
        return meetsDate;
    } else if (useDup && useDate) {
        // console.log(`[shouldHighlight] Итог meetsDuplicates || meetsDate: ${meetsDuplicates || meetsDate}`);
        return meetsDuplicates || meetsDate;
    }

    // console.log('[shouldHighlight] Итог: false');
    return false;
}

function applyHighlight(element, shouldHighlight) {
    if (!element) return;
    ensureHighlightStyle();
    if (shouldHighlight) {
        element.classList.add('universal-id-flagged');
    } else {
        element.classList.remove('universal-id-flagged');
    }
}

// ==================== УПРОЩЁННЫЕ API ЗАПРОСЫ ====================

async function getApiKey() {
    if (apiKeyLoaded) return cachedApiKey;
    return new Promise((resolve) => {
        chrome.storage.local.get(["apiKey", "my_api_key"], (res) => {
            cachedApiKey = res.apiKey || res.my_api_key || null;
            apiKeyLoaded = true;
            resolve(cachedApiKey);
        });
    });
}

// ОБНОВЛЕННАЯ функция проверки записей с учетом групп

// ==================== BULK API ДЛЯ ID ====================
// Пакетная проверка ID
async function bulkCheckIds(ids) {
    if (!Array.isArray(ids) || ids.length === 0) return {};
    try {
        const res = await apiRequest({ url: API_ENDPOINTS.BULK_CHECK, method: 'POST', body: { ids, site: SITE_NAME } });
        if (!res || typeof res !== 'object') return {};
        if (res.status !== 200) return {};
        const data = res.data || {};
        if (data && typeof data === 'object') {
            if (data.results && typeof data.results === 'object') return data.results;
            return data;
        }
        return {};
    } catch (e) {
        console.error("Ошибка пакетной проверки ID:", e);
    }
    return {};
}

// Пакетное добавление ID
async function bulkAddIds(ids, apiKey) {
    if (!Array.isArray(ids) || ids.length === 0) return false;
    try {
        const res = await apiRequest({ url: `${SERVER_URL}/api/ids/bulk-add`, method: 'POST', body: { ids, apiKey, site: SITE_NAME } });
        if (!res || res.status !== 200) return false;
        const data = res.data || {};
        return data.success === true;
    } catch (e) {
        console.error("Ошибка пакетной отправки ID:", e);
    }
    return false;
}

// Обновить заметку для ID
async function updateNoteForId(id, note, apiKey) {
    if (!id || !apiKey) return { success: false, error: 'missing id/apiKey' };
    try {
        const res = await apiRequest({ url: API_ENDPOINTS.NOTE_BY_KEY, method: 'POST', body: { id: String(id), site: SITE_NAME, note: String(note || ''), apiKey } });
        if (!res || res.status !== 200) return { success: false, error: 'http ' + (res && res.status)};
        return res.data || { success: false };
    } catch (e) {
        console.error("Ошибка обновления заметки:", e);
        return { success: false, error: String(e && e.message ? e.message : e) };
    }
}

// Одиночная проверка через bulkCheckIds
async function fetchRecordsForId(id) {
    id = String(id);
    const CACHE_TIME = 3000;
    const now = Date.now();
    if (recordsCache.has(id)) {
        const cached = recordsCache.get(id);
        if (now - cached.timestamp < CACHE_TIME) {
            return cached.records;
        }
    }
    if (updateInProgress.has(id)) {
        return recordsCache.has(id) ? recordsCache.get(id).records : [];
    }
    updateInProgress.add(id);
    try {
        // Используем bulkCheckIds для одного ID
        const results = await bulkCheckIds([id]);
        // console.log('[fetchRecordsForId] Ответ bulkCheckIds:', results);
        // console.log('[fetchRecordsForId] Ключи results:', Object.keys(results));
        // console.log('[fetchRecordsForId] id:', id);
        let allRecords = [];
        // Нормализация ключа
        const key = Object.keys(results).find(k => k == id);
        if (key && results[key]) {
            if (Array.isArray(results[key].records)) {
                allRecords = results[key].records;
            } else if (Array.isArray(results[key])) {
                allRecords = results[key];
            }
        }
        // console.log('[fetchRecordsForId] Итоговые записи для подсветки:', allRecords);
        recordsCache.set(id, {
            records: allRecords,
            timestamp: now
        });
        return allRecords;
    } catch (error) {
        console.error("Критическая ошибка загрузки записей:", error);
        return [];
    } finally {
        updateInProgress.delete(id);
    }
}

// Одиночная отправка через bulkAddIds
async function sendIdToServer(id, apiKey) {
    try {
        const success = await enqueueAddId(id, apiKey);
        return success === true;
    } catch (e) {
        console.error("Ошибка отправки ID:", id, e);
        return false;
    }
}

// ==================== UI КОМПОНЕНТЫ ====================

function createControlButtons(id, initialRecords = []) {
    const container = document.createElement("div");
    container.className = "universal-id-controls";
    container.dataset.universalId = String(id);
    Object.assign(container.style, {
        position: "absolute",
        top: "4px",
        right: "4px",
        display: "flex",
        gap: "4px",
        zIndex: "99999",
        background: "rgba(255,255,255,0.9)",
        padding: "4px",
        borderRadius: "6px",
        border: "1px solid #ddd"
    });

    const copyBtn = document.createElement("button");
    copyBtn.textContent = "📋";
    copyBtn.title = `Копировать ID: ${id}`;
    Object.assign(copyBtn.style, {
        padding: "4px 8px",
        fontSize: "12px",
        cursor: "pointer",
        border: "1px solid #99ccee",
        borderRadius: "4px",
        backgroundColor: "#e0f0ff",
        color: "#003366"
    });
    
    copyBtn.addEventListener("click", async (e) => {
        e.preventDefault();
        e.stopPropagation();
        try {
            await navigator.clipboard.writeText(id);
            copyBtn.textContent = "✅";
            setTimeout(() => copyBtn.textContent = "📋", 1500);
        } catch {
            alert("Не удалось скопировать ID");
        }
    });

    const uploadBtn = document.createElement("button");
    uploadBtn.className = "universal-upload-btn";
    updateUploadButton(uploadBtn, initialRecords, id);

    const noteBtn = document.createElement("button");
    noteBtn.textContent = "📝";
    noteBtn.title = "Заметка к ID";
    Object.assign(noteBtn.style, {
        padding: "4px 8px",
        fontSize: "12px",
        cursor: "pointer",
        border: "1px solid #cbd5e1",
        borderRadius: "4px",
        backgroundColor: "#f8fafc",
        color: "#0f172a"
    });

    noteBtn.addEventListener("click", async (e) => {
        e.preventDefault();
        e.stopPropagation();

        const apiKey = await getApiKey();
        if (!apiKey) {
            alert("API-ключ не задан. Кликните по иконке расширения и введите ключ.");
            return;
        }

        const cached = recordsCache.get(String(id));
        const cachedRecords = cached && Array.isArray(cached.records) ? cached.records : initialRecords;
        const currentRecord = Array.isArray(cachedRecords)
            ? (cachedRecords.find(r => r.site === SITE_NAME) || cachedRecords[0])
            : null;
        const currentNote = currentRecord && currentRecord.note ? String(currentRecord.note) : "";
        const newNote = prompt("Заметка для ID:", currentNote);
        if (newNote === null) return;

        const result = await updateNoteForId(id, newNote, apiKey);
        if (result && result.success) {
            // Обновим кэш заметки
            const recs = await fetchRecordsForId(id);
            recordsCache.set(String(id), { records: recs, timestamp: Date.now() });
            updateUploadButton(uploadBtn, recs, id);
        } else {
            alert("Не удалось сохранить заметку");
        }
    });

    uploadBtn.addEventListener("click", async (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        if (uploadBtn._busy) return;
        uploadBtn._busy = true;
        
        uploadBtn.textContent = "⏳";
        uploadBtn.style.backgroundColor = "#fff3cd";
        uploadBtn.style.color = "#6a4f00";
        uploadBtn.disabled = true;

        const apiKey = await getApiKey();
        if (!apiKey) {
            alert("API-ключ не задан. Кликните по иконке расширения и введите ключ.");
            uploadBtn._busy = false;
            updateUploadButton(uploadBtn, initialRecords, id);
            return;
        }

        const success = await sendIdToServer(id, apiKey);
        if (success) {
            recordsCache.delete(String(id));
            setTimeout(async () => {
                const records = await fetchRecordsForId(id);
                updateUploadButton(uploadBtn, records, id);
                const controls = uploadBtn.closest('.universal-id-controls');
                const anchor = (controls && controls.parentElement) ? controls.parentElement : uploadBtn.closest(getCurrentSelectors());
                applyHighlight(anchor, shouldHighlight(records));
                uploadBtn._busy = false;
            }, 1000);
        } else {
            uploadBtn.textContent = "⏳";
            setTimeout(async () => {
                const records = await fetchRecordsForId(id);
                updateUploadButton(uploadBtn, records, id);
                uploadBtn._busy = false;
            }, 1500);
        }
    });

    container.appendChild(copyBtn);
    container.appendChild(noteBtn);
    container.appendChild(uploadBtn);
    return container;
}

function updateUploadButton(btn, records, id) {
    const hasRecords = Array.isArray(records) && records.length > 0;
    
    btn.textContent = hasRecords ? "✅" : "➕";
    btn.disabled = false;
    
    Object.assign(btn.style, {
        padding: "4px 8px",
        fontSize: "12px",
        cursor: "pointer",
        border: hasRecords ? "1px solid #77cc88" : "1px solid #99cc99",
        borderRadius: "4px",
        backgroundColor: hasRecords ? "#2ecc71" : "#e9f9e9",
        color: hasRecords ? "white" : "#115511"
    });

    if (hasRecords) {
        // УЛУЧШЕННАЯ СТАТИСТИКА С УЧЕТОМ ГРУПП
        const recordsBySite = {};
        records.forEach(r => {
            const site = r.site || r.sourceSite || 'неизвестно';
            if (!recordsBySite[site]) {
                recordsBySite[site] = [];
            }
            recordsBySite[site].push(r);
        });

        let title = `ID: ${id}\n\n`;
        
        // Разделяем на текущий сайт и сайты из групп
        const currentSiteRecords = recordsBySite[SITE_NAME] || [];
        const groupSiteRecords = Object.entries(recordsBySite)
            .filter(([site]) => site !== SITE_NAME)
            .sort(([a], [b]) => a.localeCompare(b));
        
        // Текущий сайт
        if (currentSiteRecords.length > 0) {
            title += `ТЕКУЩИЙ САЙТ (${SITE_NAME}): ${currentSiteRecords.length} записей\n`;
            currentSiteRecords.slice(0, 2).forEach(r => {
                const user = r.added_by || 'неизвестно';
                const time = new Date(r.created_at).toLocaleString('ru-RU', {
                    timeZone: 'Europe/Moscow',
                    day: "2-digit",
                    month: "2-digit", 
                    year: "numeric",
                    hour: "2-digit",
                    minute: "2-digit"
                });
                title += `   └ ${user} (${time})\n`;
            });
            if (currentSiteRecords.length > 2) {
                title += `   └ ... и ещё ${currentSiteRecords.length - 2} записей\n`;
            }
            title += '\n';
        }
        
        // Сайты из групп
        if (groupSiteRecords.length > 0) {
            title += `НАЙДЕНО В ГРУППАХ:\n`;
            groupSiteRecords.forEach(([site, siteRecords]) => {
                title += `${site}: ${siteRecords.length} записей\n`;
                
                siteRecords.slice(0, 2).forEach(r => {
                    const user = r.added_by || 'неизвестно';
                    const time = new Date(r.created_at).toLocaleString('ru-RU', {
                        timeZone: 'Europe/Moscow',
                        day: "2-digit",
                        month: "2-digit", 
                        year: "numeric",
                        hour: "2-digit",
                        minute: "2-digit"
                    });
                    title += `   └ ${user} (${time})\n`;
                });
                
                if (siteRecords.length > 2) {
                    title += `   └ ... и ещё ${siteRecords.length - 2} записей\n`;
                }
                title += '\n';
            });
        }

        // ОБЩАЯ СТАТИСТИКА
        const totalCount = records.length;
        const currentSiteCount = currentSiteRecords.length;
        const groupsCount = totalCount - currentSiteCount;
        
        title += `СВОДКА:\n`;
        title += `   Всего записей: ${totalCount}\n`;
        title += `   На этом сайте: ${currentSiteCount}\n`;
        title += `   В группах: ${groupsCount}\n`;
        title += `   Проверено сайтов: ${Object.keys(recordsBySite).length}\n\n`;
        const noteSample = (currentSiteRecords.find(r => r.note) || {}).note;
        if (noteSample) {
            title += `Заметка: ${String(noteSample).slice(0, 120)}\n\n`;
        }
        title += `Кликните, чтобы добавить снова`;

        btn.title = title;
        
        // Изменяем цвет кнопки если найдено в группах
        if (groupsCount > 0) {
            btn.style.backgroundColor = "#f39c12"; // оранжевый для групп
            btn.style.color = "white";
            btn.style.border = "1px solid #e67e22";
        }
    } else {
        btn.title = `Добавить ID ${id} на сервер`;
    }
}

// ==================== УВЕДОМЛЕНИЯ ====================

function showNotification(message) {
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #10b981;
        color: white;
        padding: 12px 16px;
        border-radius: 8px;
        z-index: 10000;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        font-size: 14px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        max-width: 300px;
        word-wrap: break-word;
    `;
    
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 4000);
}

// ==================== ОСНОВНАЯ ЛОГИКА ====================

function attachControlsForId(el, id, matchedParser) {
    if (!el || !id) return;

    el._universalEnhanced = true;
    el._lastEnhancedTime = Date.now();

    registerElementId(el, String(id));

    try {
        const style = window.getComputedStyle(el);
        if (style.position === "static") {
            el.style.position = "relative";
        }
    } catch (styleError) {
        // ignore
    }

    const anchor = resolveButtonAnchor(el, matchedParser);
    if (!anchor) return;
    elementToAnchor.set(el, anchor);

    const existingControls = anchor.querySelector(".universal-id-controls");
    if (existingControls && existingControls._isUniversalControl) {
        const existingId = existingControls.dataset.universalId || '';
        if (existingId === String(id)) {
            return;
        }
        existingControls.remove();
    }

    const controls = createControlButtons(id);
    controls._isUniversalControl = true;
    anchor.appendChild(controls);

    updateCardStatus(String(id), el);
}

function enhanceCard(el) {
    if (!el || !el.nodeType || el.nodeType !== 1 || el._universalEnhanced) {
        return;
    }
    
    try {
        // ПРОВЕРЯЕМ СЕЛЕКТОРЫ ПАРСЕРОВ - если элемент не подходит ни под один селектор, пропускаем
        const matchedParser = getMatchingParserForElement(el);
        if (!matchedParser && siteParsers.length > 0) {
            return;
        }
        
        const id = extractIDFromElement(el);
        const useNickname = shouldUseNickname(matchedParser);
        const registeredId = getRegisteredIdForElement(el);

        if (useNickname) {
            if (!el._nicknamePending) {
                el._nicknamePending = true;
                resolveNicknameForElement(el, matchedParser).then((nickname) => {
                    el._nicknamePending = false;
                    if (nickname) {
                        attachControlsForId(el, nickname, matchedParser);
                    }
                }).catch(() => {
                    el._nicknamePending = false;
                });
            }

            if (registeredId) {
                const anchor = resolveButtonAnchor(el, matchedParser);
                const existingControls = anchor ? anchor.querySelector('.universal-id-controls') : null;
                if (!existingControls || existingControls.dataset.universalId !== String(registeredId)) {
                    attachControlsForId(el, registeredId, matchedParser);
                }
                return;
            }

            if (!registeredId && id) {
                attachControlsForId(el, id, matchedParser);
            }
            return;
        }

        if (!id) {
            return;
        }

        attachControlsForId(el, id, matchedParser);
        
    } catch (error) {
        console.error('Ошибка в enhanceCard:', error);
    }
}

async function updateCardStatus(id, element) {
    if (!id || !element) return;
    registerElementId(element, String(id));
    enqueueCheckId(String(id));
}

// ==================== НАБЛЮДАТЕЛЬ И ИНИЦИАЛИЗАЦИЯ ====================


// Обновление: пакетная обработка раз в минуту (меньше запросов, пакетные bulk вызовы)
setInterval(async () => {
    try {
        // Не работать если вкладка не активна
        if (typeof document !== 'undefined' && document.hidden) return;

        await loadSiteSettings();
        await loadSiteParsers(); // обновляет селекторы

        const currentSelectors = getCurrentSelectors();
        if (currentSelectors === '.universal-id-no-elements-found') return;

        const cards = Array.from(document.querySelectorAll(currentSelectors));
        const idsMap = new Map();
        for (const el of cards) {
            const id = getRegisteredIdForElement(el) || extractIDFromElement(el);
            if (id) {
                registerElementId(el, String(id));
                if (!idsMap.has(id)) idsMap.set(id, []);
                idsMap.get(id).push(el);
            }
        }

        const ids = Array.from(idsMap.keys());
        if (ids.length === 0) return;

        // Запрос единым bulk-вызовом
        const results = await bulkCheckIds(ids);

        // Обновляем кэш и DOM на основании ответа
        applyBulkResults(results, ids);
    } catch (e) {
        console.warn('Ошибка в пакетном обновлении карточек:', e);
    }
}, 60 * 1000); // 60s

// Обновленная обработка SPA навигации
// УЛУЧШЕННАЯ обработка SPA навигации с debounce
let spaNavigationTimeout = null;

function handleUrlChange() {
    if (spaNavigationTimeout) {
        clearTimeout(spaNavigationTimeout);
    }
    
    spaNavigationTimeout = setTimeout(() => {
        const newUrl = window.location.href;
        if (newUrl !== currentUrl) {
            currentUrl = newUrl;
            
            // Очищаем кэш и флаги с debounce
            recordsCache.clear();
            updateInProgress.clear();
            idToElements.clear();
            pendingCheckIds.clear();
            lastParsersLoad = 0;
            lastSettingsLoad = 0;
            
            // ПОЛНОЕ пересканирование с задержкой
            setTimeout(async () => {
                // Полностью сбрасываем все флаги
                document.querySelectorAll(getCurrentSelectors()).forEach(el => {
                    delete el._universalEnhanced;
                });
                
                // Удаляем ВСЕ старые контролы
                document.querySelectorAll('.universal-id-controls').forEach(control => {
                    if (control._isUniversalControl) {
                        control.remove();
                    }
                });
                
                await loadSiteSettings();
                await loadSiteParsers();
                
                // Полное пересканирование
                const currentSelectors = getCurrentSelectors();
                if (currentSelectors !== '.universal-id-no-elements-found') {
                    const cards = document.querySelectorAll(currentSelectors);
                    // Пакетная обработка для лучшей производительности
                    const batchSize = 20;
                    for (let i = 0; i < cards.length; i += batchSize) {
                        const batch = Array.from(cards).slice(i, i + batchSize);
                        batch.forEach(enhanceCard);
                        await new Promise(resolve => setTimeout(resolve, 50));
                    }
                }
                
                scanPageWithParsers();
            }, 3000); // Больше времени для загрузки страницы
        }
    }, 500); // Debounce 500ms
}


// Перехват SPA навигации
const originalPushState = history.pushState;
const originalReplaceState = history.replaceState;

history.pushState = function(...args) {
    originalPushState.apply(this, args);
    setTimeout(handleUrlChange, 100);
};

history.replaceState = function(...args) {
    originalReplaceState.apply(this, args);
    setTimeout(handleUrlChange, 100);
};

window.addEventListener('popstate', handleUrlChange);

// ==================== КОММУНИКАЦИЯ С POPUP ====================

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    // console.log('Content script получил сообщение:', request);
    
    if (request.action === 'getParsersInfo') {
        sendResponse({
            success: true,
            parsers: siteParsers,
            site: SITE_NAME,
            settings: siteSettings,
            parsersCount: siteParsers.length,
            selectors: getCurrentSelectors(),
            selectorsCount: DYNAMIC_SELECTORS.split(', ').length
        });
        return true;
    }
    
    if (request.action === 'refreshParsers') {
        lastParsersLoad = 0;
        loadSiteParsers().then(() => {
            sendResponse({ 
                success: true,
                parsers: siteParsers,
                parsersCount: siteParsers.length,
                selectors: getCurrentSelectors()
            });
        });
        return true;
    }
    
    if (request.action === 'getSelectorsInfo') {
        sendResponse({
            selectors: getCurrentSelectors(),
            selectorsList: DYNAMIC_SELECTORS.split(', '),
            parsersWithSelectors: siteParsers.filter(p => p.selectors && p.selectors.length > 0)
        });
        return true;
    }
    
    if (request.action === 'rescanPage') {
        recordsCache.clear();
        rescanPageWithNewSelectors();
        sendResponse({ success: true });
        return true;
    }

    if (request.action === 'highlightParser') {
        try {
            const parserId = Number(request.parserId);
            const parser = siteParsers.find(p => Number(p.id) === parserId);
            if (!parser) {
                sendResponse({ success: false, error: 'parser not found' });
                return true;
            }

            let elements = [];
            if (parser.selectors && Array.isArray(parser.selectors) && parser.selectors.length > 0) {
                parser.selectors.forEach(sel => {
                    try {
                        elements.push(...Array.from(document.querySelectorAll(sel)));
                    } catch (e) {
                        // ignore invalid selector
                    }
                });
            }

            // Fallback: search elements by pattern if no selectors
            if (elements.length === 0 && parser.pattern) {
                try {
                    const regex = new RegExp(parser.pattern, 'gi');
                    document.querySelectorAll('body *').forEach(el => {
                        try {
                            if (regex.test(el.outerHTML)) elements.push(el);
                        } catch (e) {}
                    });
                } catch (e) {
                    // invalid regex
                }
            }

            elements = Array.from(new Set(elements));
            let targets = elements;
            try {
                targets = elements.map(el => {
                    try {
                        return resolveButtonAnchor(el, parser) || el;
                    } catch (e) {
                        return el;
                    }
                });
            } catch (e) {
                targets = elements;
            }
            targets = Array.from(new Set(targets));

            if (targets.length > 0) {
                if (!document.getElementById('universal-id-highlight-style')) {
                    const st = document.createElement('style');
                    st.id = 'universal-id-highlight-style';
                    st.textContent = `.universal-id-highlight{outline:3px solid rgba(255,193,7,0.95); box-shadow:0 0 12px rgba(255,193,7,0.4); transition: all 0.2s ease;}`;
                    document.head.appendChild(st);
                }

                targets.forEach(el => el.classList.add('universal-id-highlight'));
                setTimeout(() => targets.forEach(el => el.classList.remove('universal-id-highlight')), 3500);
            }

            sendResponse({ success: true, matched: targets.length });
        } catch (err) {
            console.warn('Ошибка highlightParser:', err);
            sendResponse({ success: false, error: err.message });
        }
        return true;
    }

    if (request.action === 'startSelectorPicker') {
        try {
            startSelectorPicker();
            sendResponse({ success: true });
        } catch (e) {
            sendResponse({ success: false, error: e.message });
        }
        return true;
    }

    if (request.action === 'stopSelectorPicker') {
        try {
            stopSelectorPicker(true);
            sendResponse({ success: true });
        } catch (e) {
            sendResponse({ success: false, error: e.message });
        }
        return true;
    }
    
    if (request.action === 'getSiteInfo') {
        sendResponse({
            site: SITE_NAME,
            settings: siteSettings
        });
        return true;
    }
    
    if (request.action === 'refreshSettings' || request.action === 'refreshSettingsAndRescan') {
        (async () => {
            try {
                lastSettingsLoad = 0;
                lastParsersLoad = 0;
                recordsCache.clear();
                await loadSiteSettings();
                await loadSiteParsers();
                if (request.action === 'refreshSettingsAndRescan') {
                    rescanPageWithNewSelectors();
                    await quietUpdateCardStatuses();
                }
                sendResponse({ success: true });
            } catch (e) {
                sendResponse({ success: false, error: String(e && e.message ? e.message : e) });
            }
        })();
        return true;
    }
});

// ==================== ОТЛАДОЧНЫЕ ФУНКЦИИ ====================

// Функции для отладки селекторов
function debugSelectors() {
    // console.log('ДЕБАГ СЕЛЕКТОРОВ:');
    // console.log('Текущие селекторы:', getCurrentSelectors());
    // console.log('Парсеры с селекторами:', siteParsers.filter(p => p.selectors && p.selectors.length > 0));
    
    const elements = document.querySelectorAll(getCurrentSelectors());
    // console.log(`Найдено элементов: ${elements.length}`);
    
    elements.forEach((el, index) => {
        // console.log(`Элемент ${index + 1}:`, el);
        // console.log(`  HTML:`, el.outerHTML.substring(0, 200));
        
        // Проверяем какие селекторы совпадают
        const matchingSelectors = [];
        siteParsers.forEach(parser => {
            if (parser.selectors) {
                parser.selectors.forEach(selector => {
                    try {
                        if (el.matches(selector)) {
                            matchingSelectors.push(`${parser.name}: ${selector}`);
                        }
                    } catch (e) {
                        // Игнорируем ошибки селекторов
                    }
                });
            }
        });
        
        // console.log(`  Совпадающие селекторы:`, matchingSelectors);
    });
}

// Сделаем доступным для отладки
window.debugSelectors = debugSelectors;


// ==================== ИСПРАВЛЕННАЯ ИНИЦИАЛИЗАЦИЯ ====================

// Запуск
async function startInit() {
    // console.log("Universal ID Collector запущен (полностью адаптирован под модульный сервер)");

    // Загружаем данные
    await Promise.all([loadSiteSettings(), loadSiteParsers()]);

    // Инициализируем наблюдатель только если есть селекторы
    const currentSelectors = getCurrentSelectors();
    if (currentSelectors !== '.universal-id-no-elements-found') {
        // Создаем наблюдатель (используем переменную observer, объявленную в начале)
        observer = new MutationObserver((mutations) => {
            const currentSelectors = getCurrentSelectors();

            // Если селекторы пустые, не обрабатываем мутации
            if (currentSelectors === '.universal-id-no-elements-found') {
                return;
            }

            const added = [];
            mutations.forEach(mutation => {
                mutation.addedNodes.forEach(node => added.push(node));
            });
            queueNodesForEnhance(added, currentSelectors);
        });

        observer.observe(document.body, { childList: true, subtree: true });

        // Обрабатываем существующие карточки
        try {
            const cards = document.querySelectorAll(currentSelectors);
            // console.log(`Найдено ${cards.length} элементов для обработки`);
            cards.forEach(enhanceCard);
        } catch (error) {
            console.warn('Ошибка при обработке существующих карточек:', error);
        }
    } else {
        // console.log('Нет активных селекторов, наблюдатель не запущен');
    }

    // Резервное обновление (редко) — минимизируем нагрузку
    setInterval(async () => {
        try {
            if (typeof document !== 'undefined' && document.hidden) return;
            await loadSiteSettings();
            await loadSiteParsers();
        } catch (e) {
            console.warn('Ошибка в резервном обновлении парсеров/настроек:', e);
        }
    }, 5 * 60 * 1000); // 5 минут

    // Подключаемся к SSE для моментальных обновлений (если сервер поддерживает)
    (function initSSE(){
        try {
            const sseUrl = `${SERVER_URL}/api/ids/sse?site=${encodeURIComponent(SITE_NAME)}`;
            let retry = 2000;
            let fastErrorCount = 0; // count rapid failures (likely 404 / unsupported)
            function connect() {
                const connectStart = Date.now();
                const es = new EventSource(sseUrl);
                es.onmessage = async (e) => {
                    try {
                        const payload = JSON.parse(e.data);
                        if (payload && payload.event === 'id_added' && payload.data && payload.data.id) {
                            const id = String(payload.data.id);
                            recordsCache.delete(id);
                            // Обновляем только элементы с этим ID
                            cleanupDeadElements(id);
                            const els = idToElements.get(id);
                            if (els && els.size > 0) {
                                const recs = await fetchRecordsForId(id);
                                for (const el of Array.from(els)) {
                                    const btn = el.querySelector('.universal-upload-btn');
                                    if (btn) updateUploadButton(btn, recs, id);
                                    applyHighlight(el, shouldHighlight(recs));
                                }
                            }
                        }
                    } catch (err) {
                        console.warn('SSE parse error', err);
                    }
                };
                es.onerror = () => {
                    try {
                        es.close();
                    } catch (ignore) {}
                    const delta = Date.now() - connectStart;
                    // If the connection fails very quickly (e.g. immediate 404), assume SSE unsupported and stop retrying after 2 fast failures.
                    if (delta < 2000) {
                        fastErrorCount += 1;
                        if (fastErrorCount >= 2) {
                            console.warn('SSE appears unsupported on server (stopping retries)');
                            return; // stop reconnecting
                        }
                    }
                    // Otherwise back off and retry
                    setTimeout(() => { retry = Math.min(60000, retry * 1.5); connect(); }, retry);
                };
            }
            connect();
        } catch (e) {
            console.warn('SSE init failed', e);
        }
    })();

}

;(async function bootstrap() {
    try {
        const key = await getApiKey();
        if (key) {
            startInit().catch(e => console.warn('startInit failed', e));
            return;
        }
        console.log('API key not found — content script will wait until API key is set in extension.');
        const onStorage = (changes, area) => {
            if (area === 'local' && (changes.apiKey || changes.my_api_key)) {
                const newKey = (changes.apiKey && changes.apiKey.newValue) || (changes.my_api_key && changes.my_api_key.newValue) || null;
                if (newKey) {
                    try { chrome.storage.onChanged.removeListener(onStorage); } catch (e) {}
                    startInit().catch(e => console.warn('startInit failed', e));
                }
            }
        };
        chrome.storage.onChanged.addListener(onStorage);
    } catch (err) {
        console.warn('Bootstrap error:', err);
    }
})();